<?php
/**
 * Email Verification for Student Registration
 * Handles verification token and redirects to payment
 */

session_start();
require_once 'includes/functions.php';
require_once 'includes/url_helper.php';
require_once 'includes/favicon_helper.php';

$pdo = getDB();
$error = '';
$verified = false;

// Get token from URL
$token = trim($_GET['token'] ?? '');

if (empty($token)) {
    $error = "Invalid verification link. No token provided.";
} else {
    try {
        // Find application by token
        $stmt = $pdo->prepare("
            SELECT * FROM student_applications 
            WHERE email_verification_token = ?
            AND email_verified = 0
            AND application_status = 'submitted'
            AND submitted_at >= DATE_SUB(NOW(), INTERVAL 48 HOUR)
        ");
        $stmt->execute([$token]);
        $application = $stmt->fetch();
        
        if (!$application) {
            $error = "Invalid or expired verification link. The link may have already been used or expired after 48 hours.";
        } else {
            // Mark as verified
            $stmt = $pdo->prepare("
                UPDATE student_applications 
                SET email_verified = 1, 
                    email_verified_at = NOW(),
                    application_status = 'verified'
                WHERE id = ?
            ");
            $stmt->execute([$application['id']]);
            
            $verified = true;
            
            // Store application ID in session for payment
            $_SESSION['pending_application_id'] = $application['id'];
            $_SESSION['pending_payment_amount'] = $application['payment_amount'];
            $_SESSION['application_school'] = $application['academy_reference'];
            
            // Build absolute URL for payment page (auto-detects localhost vs production)
            $payment_url = generateURL('payment_for_application.php');
            
            // Redirect to payment after 3 seconds
            header("refresh:3;url=" . $payment_url);
        }
        
    } catch (Exception $e) {
        $error = "Error verifying email: " . $e->getMessage();
        error_log("Email verification error: " . $e->getMessage());
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Email Verification - Student Registration</title>
    <?php echoFavicon($application['academy_reference'] ?? null); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            padding: 20px;
        }
        .card {
            border: none;
            border-radius: 1rem;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-body text-center py-5">
                        
                        <?php if ($verified): ?>
                        <!-- Success -->
                        <div class="mb-4">
                            <i class="bi bi-check-circle display-1 text-success"></i>
                        </div>
                        <h2 class="text-success mb-3">Email Verified!</h2>
                        <p class="lead">Your email has been successfully verified.</p>
                        
                        <div class="alert alert-info mt-4">
                            <h5><i class="bi bi-cash-coin me-2"></i>Next Step: Payment</h5>
                            <p class="mb-0">You will be redirected to the payment page in 3 seconds...</p>
                            <div class="spinner-border spinner-border-sm text-primary mt-2"></div>
                        </div>
                        
                        <p class="mt-4">
                            <a href="<?= htmlspecialchars($payment_url ?? 'payment_for_application.php') ?>" class="btn btn-primary">
                                <i class="bi bi-arrow-right me-2"></i>Proceed to Payment Now
                            </a>
                        </p>
                        
                        <?php else: ?>
                        <!-- Error -->
                        <div class="mb-4">
                            <i class="bi bi-x-circle display-1 text-danger"></i>
                        </div>
                        <h2 class="text-danger mb-3">Verification Failed</h2>
                        <div class="alert alert-danger">
                            <?= htmlspecialchars($error) ?>
                        </div>
                        
                        <div class="mt-4">
                            <h6>Possible Reasons:</h6>
                            <ul class="text-start">
                                <li>The verification link has already been used</li>
                                <li>The link has expired (48 hours)</li>
                                <li>The link is invalid or corrupted</li>
                            </ul>
                        </div>
                        
                        <p class="mt-4">
                            <a href="contact.php" class="btn btn-primary">
                                <i class="bi bi-envelope me-2"></i>Contact School
                            </a>
                        </p>
                        <?php endif; ?>
                        
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>

