<?php
/**
 * Test script for monthly payment system
 * Tests database schema, email templates, cron job, and admin dashboard
 */

require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/email_templates.php';

echo "🧪 Testing Monthly Payment System\n";
echo "================================\n\n";

// Test 1: Database Schema
echo "1️⃣ Testing Database Schema...\n";
testDatabaseSchema();

// Test 2: Email Templates
echo "\n2️⃣ Testing Email Templates...\n";
testEmailTemplates();

// Test 3: Payment Status Functions
echo "\n3️⃣ Testing Payment Status Functions...\n";
testPaymentStatusFunctions();

// Test 4: Admin Dashboard Data
echo "\n4️⃣ Testing Admin Dashboard Data...\n";
testAdminDashboardData();

// Test 5: Cron Job Logic
echo "\n5️⃣ Testing Cron Job Logic...\n";
testCronJobLogic();

echo "\n🎉 All tests completed!\n";

function testDatabaseSchema() {
    try {
        $pdo = getDB();
        
        // Check if new columns exist
        $columns = [
            'last_payment_date',
            'next_payment_due', 
            'payment_cycle_start',
            'payment_reminder_sent',
            'warning_email_sent',
            'suspension_warning_sent',
            'moodle_status',
            'suspended_date'
        ];
        
        $stmt = $pdo->query("DESCRIBE students");
        $existingColumns = array_column($stmt->fetchAll(), 'Field');
        
        $missingColumns = [];
        foreach ($columns as $column) {
            if (!in_array($column, $existingColumns)) {
                $missingColumns[] = $column;
            }
        }
        
        if (empty($missingColumns)) {
            echo "✅ All required columns exist in students table\n";
        } else {
            echo "❌ Missing columns: " . implode(', ', $missingColumns) . "\n";
        }
        
        // Check payment_cycles table
        $stmt = $pdo->query("SHOW TABLES LIKE 'payment_cycles'");
        if ($stmt->rowCount() > 0) {
            echo "✅ payment_cycles table exists\n";
        } else {
            echo "❌ payment_cycles table missing\n";
        }
        
        // Check student data
        $stmt = $pdo->query("SELECT COUNT(*) as count FROM students WHERE payment_cycle_start IS NOT NULL");
        $count = $stmt->fetch()['count'];
        echo "✅ {$count} students have payment cycles initialized\n";
        
    } catch (Exception $e) {
        echo "❌ Database schema test failed: " . $e->getMessage() . "\n";
    }
}

function testEmailTemplates() {
    try {
        // Create test student data
        $testStudent = [
            'id' => 1,
            'first_name' => 'John',
            'last_name' => 'Doe',
            'email' => 'john.doe@example.com'
        ];
        
        $testSubjects = [
            ['name' => 'Mathematics', 'price' => 0.30],
            ['name' => 'English', 'price' => 0.30],
            ['name' => 'Science', 'price' => 0.30]
        ];
        
        $amountDue = 0.90;
        $paymentLink = "https://example.com/payments/pay.php?student_id=1";
        
        // Test reminder email
        $reminderEmail = getPaymentReminderEmail($testStudent, $amountDue, $paymentLink, $testSubjects);
        if (!empty($reminderEmail['subject']) && !empty($reminderEmail['html'])) {
            echo "✅ Payment reminder email template generated\n";
        } else {
            echo "❌ Payment reminder email template failed\n";
        }
        
        // Test warning email
        $warningEmail = getPaymentWarningEmail($testStudent, $amountDue, $paymentLink, $testSubjects);
        if (!empty($warningEmail['subject']) && !empty($warningEmail['html'])) {
            echo "✅ Payment warning email template generated\n";
        } else {
            echo "❌ Payment warning email template failed\n";
        }
        
        // Test suspension email
        $suspensionEmail = getSuspensionEmail($testStudent, $amountDue, $paymentLink, $testSubjects);
        if (!empty($suspensionEmail['subject']) && !empty($suspensionEmail['html'])) {
            echo "✅ Suspension email template generated\n";
        } else {
            echo "❌ Suspension email template failed\n";
        }
        
    } catch (Exception $e) {
        echo "❌ Email templates test failed: " . $e->getMessage() . "\n";
    }
}

function testPaymentStatusFunctions() {
    try {
        $pdo = getDB();
        
        // Test getStudentPaymentStatus function
        $stmt = $pdo->query("SELECT id FROM students LIMIT 1");
        $student = $stmt->fetch();
        
        if ($student) {
            $paymentStatus = getStudentPaymentStatus($student['id']);
            if (isset($paymentStatus['has_paid']) && isset($paymentStatus['has_pending'])) {
                echo "✅ getStudentPaymentStatus function works\n";
            } else {
                echo "❌ getStudentPaymentStatus function failed\n";
            }
        } else {
            echo "⚠️ No students found to test payment status\n";
        }
        
        // Test payment cycle data
        $stmt = $pdo->query("
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN payment_cycle_start IS NOT NULL THEN 1 ELSE 0 END) as with_cycles,
                SUM(CASE WHEN next_payment_due IS NOT NULL THEN 1 ELSE 0 END) as with_due_dates
            FROM students
        ");
        $stats = $stmt->fetch();
        
        echo "✅ Payment cycle stats: {$stats['with_cycles']}/{$stats['total']} students have cycles\n";
        echo "✅ Due date stats: {$stats['with_due_dates']}/{$stats['total']} students have due dates\n";
        
    } catch (Exception $e) {
        echo "❌ Payment status functions test failed: " . $e->getMessage() . "\n";
    }
}

function testAdminDashboardData() {
    try {
        $pdo = getDB();
        
        // Test the main query with payment status and Moodle status
        $sql = "SELECT s.*, g.name as grade_name,
                CASE 
                    WHEN EXISTS (SELECT 1 FROM payments p WHERE p.student_id = s.id AND p.status = 'SUCCESSFUL') THEN 'Paid'
                    WHEN EXISTS (SELECT 1 FROM payments p WHERE p.student_id = s.id AND p.status IN ('PENDING', 'FAILED')) THEN 'Pending'
                    ELSE 'Unpaid'
                END as payment_status,
                s.moodle_status
                FROM students s JOIN grades g ON s.grade_id = g.id
                LIMIT 5";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute();
        $students = $stmt->fetchAll();
        
        if (!empty($students)) {
            echo "✅ Admin dashboard query works\n";
            echo "✅ Sample student data:\n";
            foreach ($students as $student) {
                echo "   - {$student['first_name']} {$student['last_name']}: Payment={$student['payment_status']}, Moodle={$student['moodle_status']}\n";
            }
        } else {
            echo "❌ Admin dashboard query failed\n";
        }
        
    } catch (Exception $e) {
        echo "❌ Admin dashboard data test failed: " . $e->getMessage() . "\n";
    }
}

function testCronJobLogic() {
    try {
        $pdo = getDB();
        
        // Test date calculations
        $today = date('Y-m-d');
        $reminderDate = date('Y-m-d', strtotime('+26 days', strtotime($today)));
        $warningDate = date('Y-m-d', strtotime('+30 days', strtotime($today)));
        $suspensionDate = date('Y-m-d', strtotime('+33 days', strtotime($today)));
        
        echo "✅ Date calculations work:\n";
        echo "   - Today: $today\n";
        echo "   - Reminder date: $reminderDate\n";
        echo "   - Warning date: $warningDate\n";
        echo "   - Suspension date: $suspensionDate\n";
        
        // Test student queries for different stages
        $stages = [
            'reminder' => $reminderDate,
            'warning' => $warningDate,
            'suspension' => $suspensionDate
        ];
        
        foreach ($stages as $stage => $date) {
            $sql = "SELECT COUNT(*) as count FROM students WHERE next_payment_due = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$date]);
            $count = $stmt->fetch()['count'];
            echo "✅ Students due for $stage on $date: $count\n";
        }
        
        // Test payment cycle reset logic
        $resetSql = "
            UPDATE students 
            SET 
                last_payment_date = NOW(),
                next_payment_due = DATE_ADD(NOW(), INTERVAL 30 DAY),
                payment_reminder_sent = 0,
                warning_email_sent = 0,
                suspension_warning_sent = 0,
                moodle_status = 'active',
                suspended_date = NULL
            WHERE id = 1
        ";
        
        $stmt = $pdo->prepare($resetSql);
        $stmt->execute();
        echo "✅ Payment cycle reset logic works\n";
        
    } catch (Exception $e) {
        echo "❌ Cron job logic test failed: " . $e->getMessage() . "\n";
    }
}
?>
