<?php
/**
 * Create Safe .htaccess Version
 * 
 * Generates a compatible .htaccess file that works with both Apache 2.2 and 2.4+
 * This version uses conditional checks to prevent 500 errors
 */

header('Content-Type: text/html; charset=utf-8');

// Get RewriteBase from current installation
$scriptPath = $_SERVER['SCRIPT_NAME'];
$basePath = dirname(dirname($scriptPath));
$rewriteBase = $basePath === '/' ? '/' : $basePath . '/';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create'])) {
    $htaccessContent = generateSafeHtaccess($_POST['rewrite_base'] ?? $rewriteBase);
    $backupCreated = false;
    
    // Backup existing .htaccess if it exists
    $htaccessPath = __DIR__ . '/../.htaccess';
    $backupPath = __DIR__ . '/../.htaccess.backup.' . date('YmdHis');
    
    if (file_exists($htaccessPath)) {
        if (copy($htaccessPath, $backupPath)) {
            $backupCreated = true;
        }
    }
    
    // Write new .htaccess
    $written = file_put_contents($htaccessPath, $htaccessContent);
    
    if ($written !== false) {
        $success = true;
        $message = "✅ .htaccess file created successfully!";
        if ($backupCreated) {
            $message .= " (Backup saved as: " . basename($backupPath) . ")";
        }
    } else {
        $success = false;
        $message = "❌ Failed to write .htaccess file. Check file permissions.";
    }
}

function generateSafeHtaccess($rewriteBase) {
    return <<<HTACCESS
# =====================================================
# MULTI-TENANT SCHOOL SYSTEM - SAFE CLEAN URLS & SECURITY
# =====================================================
# Version: 2.0-Safe
# Compatible with Apache 2.2 and 2.4+
# =====================================================

<IfModule mod_rewrite.c>
    RewriteEngine On
    RewriteBase {$rewriteBase}
    
    # =====================================================
    # OPTIONAL: HTTPS ENFORCEMENT
    # =====================================================
    # Uncomment the following 2 lines to enforce HTTPS
    # RewriteCond %{HTTPS} off
    # RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
    
    # =====================================================
    # PREVENT DIRECT PHP FILE ACCESS (Optional)
    # =====================================================
    # Uncomment if you want to force clean URLs
    # RewriteCond %{THE_REQUEST} \s/+([^?\s]*?)(\.php)[?\s] [NC]
    # RewriteRule ^ /%1? [R=301,L]
    
    # =====================================================
    # TENANT ISOLATION - SCHOOL-SPECIFIC URLS
    # =====================================================
    
    # Student Registration: /school-code/register
    RewriteRule ^([a-zA-Z0-9_-]+)/register/?$ register_student.php?school=$1 [L,QSA]
    
    # Parent Registration: /school-code/parent/register
    RewriteRule ^([a-zA-Z0-9_-]+)/parent/register/?$ parent_register.php?school=$1 [L,QSA]
    
    # Parent Login: /school-code/parent/login
    RewriteRule ^([a-zA-Z0-9_-]+)/parent/login/?$ parent_login.php?school=$1 [L,QSA]
    
    # Student Login: /school-code/login
    RewriteRule ^([a-zA-Z0-9_-]+)/login/?$ student_login.php?school=$1 [L,QSA]
    
    # School Info: /school-code/info
    RewriteRule ^([a-zA-Z0-9_-]+)/info/?$ index.php?school=$1 [L,QSA]
    
    # Tenant Admin Access: /school/{code}/admin/{page}
    RewriteRule ^school/([a-zA-Z0-9_-]+)/admin/([a-zA-Z0-9_/-]+)/?$ admin/$2.php?school=$1 [L,QSA]
    
    # Tenant Super Admin Access: /school/{code}/super_admin/{page}
    RewriteRule ^school/([a-zA-Z0-9_-]+)/super_admin/([a-zA-Z0-9_/-]+)/?$ super_admin/$2.php?school=$1 [L,QSA]
    
    # =====================================================
    # CLEAN URLS - REMOVE .PHP EXTENSIONS
    # =====================================================
    
    # Admin panel clean URLs
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteRule ^admin/([a-zA-Z0-9_/-]+)/?$ admin/$1.php [L,QSA]
    
    # Super admin panel clean URLs
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteRule ^super_admin/([a-zA-Z0-9_/-]+)/?$ super_admin/$1.php [L,QSA]
    
    # Parent panel clean URLs
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteRule ^parent/([a-zA-Z0-9_/-]+)/?$ parent/$1.php [L,QSA]
    
    # Student panel clean URLs
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteRule ^student/([a-zA-Z0-9_/-]+)/?$ student/$1.php [L,QSA]
    
    # Root-level clean URLs
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteRule ^([a-zA-Z0-9_/-]+)/?$ $1.php [L,QSA]
    
    # Remove trailing slash from files
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_URI} (.+)/$
    RewriteRule ^ %1 [R=301,L]
    
    # =====================================================
    # PROTECT SENSITIVE DIRECTORIES
    # =====================================================
    
    # Block access to vendor directory
    RewriteCond %{REQUEST_URI} ^.*/vendor/ [NC]
    RewriteRule . - [F,L]
    
    # Block access to database directory
    RewriteCond %{REQUEST_URI} ^.*/database/ [NC]
    RewriteRule . - [F,L]
    
    # Block access to logs directory
    RewriteCond %{REQUEST_URI} ^.*/logs/ [NC]
    RewriteRule . - [F,L]
    
</IfModule>

# =====================================================
# SECURITY - DIRECTORY BROWSING
# =====================================================
Options -Indexes

# =====================================================
# SECURITY - SENSITIVE FILES PROTECTION
# =====================================================
# Apache 2.4+ compatible
<FilesMatch "\.(sql|log|bak|backup|old|orig|save|swp|tmp|temp|ini|env|htaccess|htpasswd|git|svn|DS_Store)$">
    <IfModule mod_authz_core.c>
        # Apache 2.4+
        Require all denied
    </IfModule>
    <IfModule !mod_authz_core.c>
        # Apache 2.2 fallback
        Order allow,deny
        Deny from all
    </IfModule>
</FilesMatch>

# =====================================================
# SECURITY - PROTECT CONFIG FILES
# =====================================================
<FilesMatch "^(config|database|settings)\.php$">
    <IfModule mod_authz_core.c>
        Require all denied
    </IfModule>
    <IfModule !mod_authz_core.c>
        Order allow,deny
        Deny from all
    </IfModule>
</FilesMatch>

# =====================================================
# SECURITY - PROTECT INCLUDES DIRECTORY
# =====================================================
<IfModule mod_rewrite.c>
    RewriteCond %{REQUEST_URI} ^.*/includes/ [NC]
    RewriteCond %{REQUEST_URI} !\.(css|js|png|jpg|jpeg|gif|svg|ico|woff|woff2|ttf|eot)$ [NC]
    RewriteRule . - [F,L]
</IfModule>

# =====================================================
# SECURITY - UPLOAD DIRECTORY PROTECTION
# =====================================================
<DirectoryMatch "^.*/(uploads|tenants)/">
    <FilesMatch "\.(php|phtml|php3|php4|php5|pl|py|jsp|asp|sh|cgi)$">
        <IfModule mod_authz_core.c>
            Require all denied
        </IfModule>
        <IfModule !mod_authz_core.c>
            Order allow,deny
            Deny from all
        </IfModule>
    </FilesMatch>
</DirectoryMatch>

# =====================================================
# SECURITY - HEADERS (if mod_headers is available)
# =====================================================
<IfModule mod_headers.c>
    Header always set X-Frame-Options "SAMEORIGIN"
    Header always set X-XSS-Protection "1; mode=block"
    Header always set X-Content-Type-Options "nosniff"
    Header always set Referrer-Policy "strict-origin-when-cross-origin"
</IfModule>

# =====================================================
# PERFORMANCE - CACHING (if mod_expires is available)
# =====================================================
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpg "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType image/svg+xml "access plus 1 year"
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType image/x-icon "access plus 1 year"
</IfModule>

# =====================================================
# CHARACTER ENCODING
# =====================================================
AddDefaultCharset UTF-8

# =====================================================
# END
# =====================================================
HTACCESS;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Safe .htaccess</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            max-width: 800px;
            margin: 50px auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #007bff;
            border-bottom: 3px solid #007bff;
            padding-bottom: 10px;
        }
        .form-group {
            margin: 20px 0;
        }
        label {
            display: block;
            font-weight: bold;
            margin-bottom: 5px;
        }
        input[type="text"] {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 1em;
        }
        .help-text {
            font-size: 0.9em;
            color: #666;
            margin-top: 5px;
        }
        .btn {
            display: inline-block;
            padding: 12px 24px;
            background: #28a745;
            color: white;
            text-decoration: none;
            border: none;
            border-radius: 4px;
            font-size: 1em;
            cursor: pointer;
        }
        .btn:hover {
            background: #218838;
        }
        .alert {
            padding: 15px;
            border-radius: 4px;
            margin: 20px 0;
        }
        .alert-success {
            background: #d4edda;
            border-left: 4px solid #28a745;
            color: #155724;
        }
        .alert-danger {
            background: #f8d7da;
            border-left: 4px solid #dc3545;
            color: #721c24;
        }
        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: monospace;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔧 Create Safe .htaccess Version</h1>
        <p>This tool generates a compatible .htaccess file that works with both Apache 2.2 and 2.4+.</p>
        
        <?php if (isset($success)): ?>
            <div class="alert <?= $success ? 'alert-success' : 'alert-danger' ?>">
                <?= $message ?>
            </div>
            <?php if ($success): ?>
                <p><strong>Next steps:</strong></p>
                <ol>
                    <li>Test your website: <a href="../">Go to homepage</a></li>
                    <li>Verify clean URLs work: <a href="../admin/dashboard">Test admin dashboard</a></li>
                    <li>If still having issues, check Apache error logs</li>
                </ol>
            <?php endif; ?>
        <?php else: ?>
            <form method="POST">
                <div class="form-group">
                    <label for="rewrite_base">RewriteBase Path:</label>
                    <input type="text" id="rewrite_base" name="rewrite_base" 
                           value="<?= htmlspecialchars($rewriteBase) ?>" required>
                    <div class="help-text">
                        This should match your installation path.<br>
                        Detected: <code><?= htmlspecialchars($rewriteBase) ?></code><br>
                        Examples: <code>/Multi-Tanent/</code> or <code>/</code> for root
                    </div>
                </div>
                
                <div class="form-group">
                    <button type="submit" name="create" class="btn">Create Safe .htaccess</button>
                    <a href="diagnose_htaccess_error.php" class="btn" style="background: #6c757d;">← Back to Diagnostics</a>
                </div>
            </form>
            
            <div style="margin-top: 30px; padding: 15px; background: #fff3cd; border-left: 4px solid #ffc107; border-radius: 4px;">
                <strong>⚠️ Important:</strong>
                <ul style="margin-top: 10px;">
                    <li>This will backup your existing .htaccess file</li>
                    <li>The new version is compatible with both Apache 2.2 and 2.4+</li>
                    <li>All optional features are wrapped in <code>&lt;IfModule&gt;</code> checks</li>
                    <li>Legacy Apache 2.2 directives are included as fallbacks</li>
                </ul>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>

