<?php
/**
 * Test Payment Reminder System
 * Run this to test the payment reminder functionality
 * 
 * Usage: php test_payment_reminder.php
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/includes/payment_reminder_manager.php';
require_once __DIR__ . '/includes/moodle_suspension_handler.php';

echo "\n";
echo "========================================\n";
echo "Payment Reminder System Test\n";
echo "========================================\n\n";

try {
    $pdo = getDB();
    $manager = new PaymentReminderManager();
    $suspension_handler = new MoodleSuspensionHandler();
    
    // Get a test student
    $stmt = $pdo->query("
        SELECT s.*, sps.*
        FROM students s
        INNER JOIN student_payment_schedules sps ON s.id = sps.student_id
        LIMIT 1
    ");
    $student = $stmt->fetch();
    
    if (!$student) {
        echo "✗ No students found with payment schedules.\n";
        echo "Run initialize_payment_system.php first.\n\n";
        exit(1);
    }
    
    echo "Testing with student: {$student['full_name']} (ID: {$student['id']})\n\n";
    
    // =====================================================
    // TEST 1: Generate Payment Link
    // =====================================================
    echo "Test 1: Payment Link Generation\n";
    echo "-----------------------------------\n";
    
    // Use reflection to access private method (for testing)
    $reflection = new ReflectionClass($manager);
    $method = $reflection->getMethod('generatePaymentLink');
    $method->setAccessible(true);
    $payment_data = $method->invoke($manager, $student['id'], 350.00);
    
    echo "✓ Payment link generated:\n";
    echo "  URL: {$payment_data['payment_link']}\n";
    echo "  Token: " . substr($payment_data['token'], 0, 50) . "...\n";
    echo "  Expires: {$payment_data['expires_at']}\n\n";
    
    // Test token decryption
    $decrypt_result = $manager->decryptToken($payment_data['token']);
    if ($decrypt_result['success']) {
        echo "✓ Token decryption successful\n";
        echo "  Student ID: {$decrypt_result['data']['student_id']}\n";
        echo "  Amount: {$decrypt_result['data']['amount']}\n\n";
    } else {
        echo "✗ Token decryption failed: {$decrypt_result['error']}\n\n";
    }
    
    // =====================================================
    // TEST 2: Send Payment Reminder
    // =====================================================
    echo "Test 2: Send Payment Reminder\n";
    echo "-----------------------------------\n";
    
    $reminder_result = $manager->sendPaymentReminder($student['id'], 'initial_reminder');
    
    if ($reminder_result['success']) {
        echo "✓ Reminder sent successfully!\n";
        echo "  Reminder ID: {$reminder_result['reminder_id']}\n";
        
        if (isset($reminder_result['notifications']['email'])) {
            echo "  Email: " . ($reminder_result['notifications']['email']['success'] ? '✓ Sent' : '✗ Failed') . "\n";
        }
        if (isset($reminder_result['notifications']['sms'])) {
            echo "  SMS: " . ($reminder_result['notifications']['sms']['success'] ? '✓ Sent' : '✗ Failed') . "\n";
        }
    } else {
        echo "✗ Reminder failed: {$reminder_result['error']}\n";
    }
    echo "\n";
    
    // =====================================================
    // TEST 3: Check Views
    // =====================================================
    echo "Test 3: Database Views\n";
    echo "-----------------------------------\n";
    
    $views = [
        'v_students_need_reminder',
        'v_students_need_warning',
        'v_students_to_suspend'
    ];
    
    foreach ($views as $view) {
        $stmt = $pdo->query("SELECT COUNT(*) as count FROM {$view}");
        $count = $stmt->fetch()['count'];
        echo "✓ {$view}: {$count} students\n";
    }
    echo "\n";
    
    // =====================================================
    // TEST 4: Moodle Suspension (OPTIONAL - Comment out if not testing)
    // =====================================================
    echo "Test 4: Moodle Suspension API\n";
    echo "-----------------------------------\n";
    echo "⚠ Skipping Moodle suspension test (uncomment to test)\n";
    /*
    if ($student['moodle_user_id']) {
        // Get current status
        $status = $suspension_handler->getSuspensionStatus($student['id']);
        echo "Current Moodle status:\n";
        echo "  Suspended: " . ($status['suspended'] ? 'Yes' : 'No') . "\n";
        
        // Test suspend
        echo "\nTesting suspension...\n";
        $suspend_result = $suspension_handler->suspendUser($student['id'], 'test');
        echo "  Result: " . ($suspend_result['success'] ? '✓ Success' : '✗ Failed: ' . $suspend_result['error']) . "\n";
        
        // Test unsuspend
        echo "\nTesting unsuspension...\n";
        $unsuspend_result = $suspension_handler->unsuspendUser($student['id']);
        echo "  Result: " . ($unsuspend_result['success'] ? '✓ Success' : '✗ Failed: ' . $unsuspend_result['error']) . "\n";
    } else {
        echo "⚠ Student not synced to Moodle yet\n";
    }
    */
    echo "\n";
    
    // =====================================================
    // SUMMARY
    // =====================================================
    echo "========================================\n";
    echo "✅ ALL TESTS COMPLETED!\n";
    echo "========================================\n\n";
    
    echo "System Status:\n";
    echo "✓ Database tables: OK\n";
    echo "✓ Payment link generation: OK\n";
    echo "✓ Token encryption/decryption: OK\n";
    echo "✓ Reminder sending: " . ($reminder_result['success'] ? 'OK' : 'FAILED') . "\n";
    echo "✓ Database views: OK\n\n";
    
    echo "Next Steps:\n";
    echo "1. Configure cron job (see PAYMENT_REMINDER_SYSTEM_SETUP_GUIDE.md)\n";
    echo "2. Access admin dashboard: admin/settings/payment_settings.php\n";
    echo "3. Review student payments: admin/settings/manage_student_payments.php\n";
    echo "4. Monitor logs: logs/payment_reminder_cron.log\n\n";
    
    echo "🎉 Payment reminder system is ready!\n\n";
    
} catch (Exception $e) {
    echo "\n✗ CRITICAL ERROR: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n\n";
    exit(1);
}

