<?php
/**
 * Create School Admin
 * Super Admin creates admin accounts for specific schools
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
requireAdminLogin();

if (!isSuperAdmin()) {
    header('Location: ../../admin/dashboard.php');
    exit;
}

$pdo = getDB();
$errors = [];
$success = false;

// Get all schools for dropdown
$stmt = $pdo->query("SELECT id, academy_name, reference_code FROM academy_references WHERE is_active = 1 ORDER BY academy_name");
$schools = $stmt->fetchAll();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    $school_id = (int)($_POST['school_id'] ?? 0);
    $momo_number = trim($_POST['momo_number'] ?? '');
    
    // Validation
    if (empty($username)) $errors[] = 'Username is required';
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = 'Valid email is required';
    if (empty($password) || strlen($password) < 8) $errors[] = 'Password must be at least 8 characters';
    if ($password !== $confirm_password) $errors[] = 'Passwords do not match';
    if ($school_id <= 0) $errors[] = 'Please select a school';
    
    if (empty($errors)) {
        try {
            // Get school reference code
            $stmt = $pdo->prepare("SELECT reference_code FROM academy_references WHERE id = ?");
            $stmt->execute([$school_id]);
            $school = $stmt->fetch();
            
            if (!$school) {
                throw new Exception('School not found');
            }
            
            // Check if username already exists
            $stmt = $pdo->prepare("SELECT id FROM admins WHERE username = ?");
            $stmt->execute([$username]);
            if ($stmt->fetch()) {
                throw new Exception('Username already exists');
            }
            
            // Check if email already exists
            $stmt = $pdo->prepare("SELECT id FROM admins WHERE email = ?");
            $stmt->execute([$email]);
            if ($stmt->fetch()) {
                throw new Exception('Email already exists');
            }
            
            // Create admin account
            $password_hash = password_hash($password, PASSWORD_DEFAULT);
            
            $stmt = $pdo->prepare("
                INSERT INTO admins (
                    username, 
                    password_hash, 
                    email,
                    academy_reference,
                    momo_number,
                    is_active,
                    created_at
                ) VALUES (?, ?, ?, ?, ?, 1, NOW())
            ");
            $stmt->execute([
                $username,
                $password_hash,
                $email,
                $school['reference_code'],
                $momo_number
            ]);
            
            $admin_id = $pdo->lastInsertId();
            
            // Create wallet for admin
            $stmt = $pdo->prepare("
                INSERT INTO admin_wallet (admin_id, balance, total_earnings, withdrawn)
                VALUES (?, 0.00, 0.00, 0.00)
            ");
            $stmt->execute([$admin_id]);
            
            $success = true;
            $successMessage = "School admin created successfully! Username: $username";
            
            // Log activity
            error_log("Super admin created school admin: $username for {$school['reference_code']}");
            
        } catch (Exception $e) {
            $errors[] = 'Error creating admin: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create School Admin - Super Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 40px 0;
        }
        .card {
            border: none;
            border-radius: 1rem;
            box-shadow: 0 10px 40px rgba(0,0,0,0.15);
        }
        .card-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 1rem 1rem 0 0 !important;
            padding: 2rem;
        }
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
        }
        .is-invalid {
            border-color: #dc3545 !important;
            box-shadow: 0 0 0 0.2rem rgba(220, 53, 69, 0.25) !important;
        }
        .form-control:focus, .form-select:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header text-center">
                        <h2 class="mb-2"><i class="bi bi-person-plus-fill"></i> Create School Admin</h2>
                        <p class="mb-0">Create an admin account for a specific school</p>
                    </div>
                    <div class="card-body p-4">
                        <?php if (!empty($errors)): ?>
                            <div class="alert alert-danger">
                                <i class="bi bi-exclamation-triangle-fill me-2"></i>
                                <strong>Please fix the following errors:</strong>
                                <ul class="mb-0 mt-2">
                                    <?php foreach ($errors as $error): ?>
                                        <li><?= htmlspecialchars($error) ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($success): ?>
                            <div class="alert alert-success">
                                <i class="bi bi-check-circle-fill me-2"></i>
                                <?= htmlspecialchars($successMessage) ?>
                            </div>
                            <div class="d-grid gap-2">
                                <a href="list.php" class="btn btn-primary">
                                    <i class="bi bi-people me-2"></i>View All Admins
                                </a>
                                <a href="create.php" class="btn btn-outline-secondary">
                                    <i class="bi bi-plus-circle me-2"></i>Create Another Admin
                                </a>
                                <a href="../dashboard.php" class="btn btn-outline-secondary">
                                    <i class="bi bi-arrow-left me-2"></i>Back to Dashboard
                                </a>
                            </div>
                        <?php else: ?>
                        
                        <form method="POST" action="">
                            <!-- School Selection -->
                            <div class="mb-4">
                                <label for="school_id" class="form-label">
                                    <i class="bi bi-building me-1"></i>Assign to School <span class="text-danger">*</span>
                                </label>
                                <select class="form-select form-select-lg" id="school_id" name="school_id" required>
                                    <option value="">-- Select School --</option>
                                    <?php foreach ($schools as $school): ?>
                                        <option value="<?= $school['id'] ?>" <?= (isset($_POST['school_id']) && $_POST['school_id'] == $school['id']) ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($school['academy_name']) ?> (<?= htmlspecialchars($school['reference_code']) ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <small class="text-muted">This admin will only have access to the selected school</small>
                            </div>
                            
                            <hr class="my-4">
                            
                            <!-- Admin Details -->
                            <h5 class="mb-3"><i class="bi bi-person-badge me-2"></i>Admin Details</h5>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="username" class="form-label">
                                        Username <span class="text-danger">*</span>
                                    </label>
                                    <input type="text" class="form-control" id="username" name="username" 
                                           value="<?= htmlspecialchars($_POST['username'] ?? '') ?>" 
                                           pattern="[a-zA-Z0-9_]{3,50}" required>
                                    <small class="text-muted">3-50 characters, letters, numbers, underscore only</small>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="email" class="form-label">
                                        Email Address <span class="text-danger">*</span>
                                    </label>
                                    <input type="email" class="form-control" id="email" name="email" 
                                           value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" required>
                                    <small class="text-muted">For account recovery and notifications</small>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="password" class="form-label">
                                        Password <span class="text-danger">*</span>
                                    </label>
                                    <input type="password" class="form-control" id="password" name="password" 
                                           minlength="8" required>
                                    <small class="text-muted">Minimum 8 characters</small>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="confirm_password" class="form-label">
                                        Confirm Password <span class="text-danger">*</span>
                                    </label>
                                    <input type="password" class="form-control" id="confirm_password" name="confirm_password" 
                                           minlength="8" required>
                                </div>
                            </div>
                            
                            <div class="mb-4">
                                <label for="momo_number" class="form-label">
                                    <i class="bi bi-phone me-1"></i>MoMo Number (Optional)
                                </label>
                                <div class="input-group">
                                    <span class="input-group-text">+268</span>
                                    <input type="tel" class="form-control" id="momo_number" name="momo_number" 
                                           value="<?= htmlspecialchars($_POST['momo_number'] ?? '') ?>" 
                                           placeholder="7687 3207">
                                </div>
                                <small class="text-muted">For receiving platform fees/commissions</small>
                            </div>
                            
                            <div class="alert alert-info">
                                <i class="bi bi-info-circle me-2"></i>
                                <strong>Access Permissions:</strong><br>
                                This admin will have full access to their assigned school only:
                                <ul class="mb-0 mt-2">
                                    <li>View and manage students</li>
                                    <li>Process payments</li>
                                    <li>Configure school settings</li>
                                    <li>View reports and analytics</li>
                                    <li>Cannot access other schools</li>
                                    <li>Cannot access super admin features</li>
                                </ul>
                            </div>
                            
                            <div class="d-grid gap-2">
                                <button type="submit" class="btn btn-primary btn-lg" id="submit_btn">
                                    <i class="bi bi-person-check me-2"></i>Create School Admin
                                </button>
                                <a href="../dashboard.php" class="btn btn-outline-secondary">
                                    <i class="bi bi-x-circle me-2"></i>Cancel
                                </a>
                            </div>
                            
                            <div class="text-center mt-3">
                                <small class="text-muted">
                                    <i class="bi bi-shield-check me-1"></i>
                                    All fields marked with <span class="text-danger">*</span> are required
                                </small>
                            </div>
                        </form>
                        
                        <?php endif; ?>
                    </div>
                </div>
                
                <div class="text-center mt-3">
                    <a href="../dashboard.php" class="text-white text-decoration-none">
                        <i class="bi bi-arrow-left me-2"></i>Back to Super Admin Dashboard
                    </a>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Comprehensive form validation
        document.querySelector('form').addEventListener('submit', function(e) {
            const errors = [];
            
            // Check school selection
            const schoolId = document.getElementById('school_id').value;
            if (!schoolId || schoolId == '') {
                errors.push('Please select a school');
                document.getElementById('school_id').classList.add('is-invalid');
            } else {
                document.getElementById('school_id').classList.remove('is-invalid');
            }
            
            // Check email
            const email = document.getElementById('email').value.trim();
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!email) {
                errors.push('Email is required');
                document.getElementById('email').classList.add('is-invalid');
            } else if (!emailRegex.test(email)) {
                errors.push('Please enter a valid email address');
                document.getElementById('email').classList.add('is-invalid');
            } else {
                document.getElementById('email').classList.remove('is-invalid');
            }
            
            // Check username
            const username = document.getElementById('username').value.trim();
            const usernameRegex = /^[a-zA-Z0-9_]{3,50}$/;
            if (!username) {
                errors.push('Username is required');
                document.getElementById('username').classList.add('is-invalid');
            } else if (!usernameRegex.test(username)) {
                errors.push('Username must be 3-50 characters (letters, numbers, underscore only)');
                document.getElementById('username').classList.add('is-invalid');
            } else {
                document.getElementById('username').classList.remove('is-invalid');
            }
            
            // Check password
            const password = document.getElementById('password').value;
            const confirmPassword = document.getElementById('confirm_password').value;
            
            if (!password) {
                errors.push('Password is required');
                document.getElementById('password').classList.add('is-invalid');
            } else if (password.length < 8) {
                errors.push('Password must be at least 8 characters');
                document.getElementById('password').classList.add('is-invalid');
            } else {
                document.getElementById('password').classList.remove('is-invalid');
            }
            
            if (!confirmPassword) {
                errors.push('Please confirm your password');
                document.getElementById('confirm_password').classList.add('is-invalid');
            } else if (password !== confirmPassword) {
                errors.push('Passwords do not match!');
                document.getElementById('confirm_password').classList.add('is-invalid');
            } else {
                document.getElementById('confirm_password').classList.remove('is-invalid');
            }
            
            // If there are errors, prevent submission and show alert
            if (errors.length > 0) {
                e.preventDefault();
                
                let errorMessage = '❌ Please fix the following errors:\n\n';
                errors.forEach((error, index) => {
                    errorMessage += `${index + 1}. ${error}\n`;
                });
                
                alert(errorMessage);
                
                // Scroll to first error field
                const firstInvalid = document.querySelector('.is-invalid');
                if (firstInvalid) {
                    firstInvalid.scrollIntoView({ behavior: 'smooth', block: 'center' });
                    firstInvalid.focus();
                }
            } else {
                // Show loading state on button
                const submitBtn = document.getElementById('submit_btn');
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Creating Admin...';
            }
        });
        
        // Real-time validation feedback
        document.getElementById('email').addEventListener('blur', function() {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (this.value && !emailRegex.test(this.value)) {
                this.classList.add('is-invalid');
            } else {
                this.classList.remove('is-invalid');
            }
        });
        
        document.getElementById('username').addEventListener('blur', function() {
            const usernameRegex = /^[a-zA-Z0-9_]{3,50}$/;
            if (this.value && !usernameRegex.test(this.value)) {
                this.classList.add('is-invalid');
            } else {
                this.classList.remove('is-invalid');
            }
        });
        
        document.getElementById('confirm_password').addEventListener('input', function() {
            const password = document.getElementById('password').value;
            if (this.value && this.value !== password) {
                this.classList.add('is-invalid');
            } else {
                this.classList.remove('is-invalid');
            }
        });
    </script>
</body>
</html>
