<?php
/**
 * Student Login Page
 * Allows students to login with school-specific branding
 * Supports clean URLs: /school-code/login
 */

session_start();
require_once 'includes/functions.php';

// Get school from URL parameter or clean URL path
$school_code = trim($_GET['school'] ?? '');

// If not in query string, try to get from clean URL path
// Clean URL format: /school-code/login
if (empty($school_code)) {
    $uri = $_SERVER['REQUEST_URI'] ?? '';
    // Extract school code from path like /Multi-Tanent/kine/login
    if (preg_match('#/([a-zA-Z0-9_-]+)/login#', $uri, $matches)) {
        $school_code = strtoupper($matches[1]);
    }
}

// Load school branding if school code provided
$school_branding = null;
$pdo = getDB();

if ($school_code) {
    $stmt = $pdo->prepare("
        SELECT ar.reference_code, ar.academy_name, ar.contact_email,
               ts.setting_key, ts.setting_value
        FROM academy_references ar
        LEFT JOIN tenant_settings ts ON ar.reference_code = ts.academy_reference
        WHERE ar.reference_code = ? AND ar.is_active = 1
    ");
    $stmt->execute([$school_code]);
    $results = $stmt->fetchAll();
    
    if (!empty($results)) {
        $school_branding = [
            'code' => $results[0]['reference_code'],
            'name' => $results[0]['academy_name'],
            'contact_email' => $results[0]['contact_email']
        ];
        foreach ($results as $row) {
            if ($row['setting_key']) {
                $school_branding[$row['setting_key']] = $row['setting_value'];
            }
        }
    }
}

// Redirect if already logged in
if (isStudentLoggedIn()) {
    header('Location: student/dashboard.php');
    exit;
}

$error = '';
$success = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate CSRF token
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid security token. Please try again.';
    } else {
        $username = trim($_POST['username'] ?? '');
        $password = trim($_POST['password'] ?? '');
        
        if (empty($username) || empty($password)) {
            $error = 'Please enter both username and password.';
        } else {
            // Attempt login (with school context if available)
            if (studentLogin($username, $password, $school_code)) {
                // Login successful - redirect to student dashboard
                header('Location: student/dashboard.php');
                exit;
            } else {
                $error = 'Invalid username or password.';
            }
        }
    }
}

// Get school logo
$school_logo = 'images/img-melanesys-logo.svg'; // Default logo
if ($school_branding && !empty($school_branding['logo_url'])) {
    $school_logo = $school_branding['logo_url'];
}

// Get school colors
$primary_color = $school_branding['primary_color'] ?? '#0d6efd';
$secondary_color = $school_branding['secondary_color'] ?? '#6c757d';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student Login<?php if ($school_branding): ?> - <?= htmlspecialchars($school_branding['name']) ?><?php endif; ?></title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="images/img-melanesys-logo.svg">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    
    <style>
        :root {
            --primary-color: <?= htmlspecialchars($primary_color) ?>;
            --secondary-color: <?= htmlspecialchars($secondary_color) ?>;
        }
        
        body {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
            min-height: 100vh;
        }
        
        .login-card {
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }
        
        .school-logo {
            max-width: 120px;
            max-height: 120px;
            object-fit: contain;
        }
        
        .btn-primary {
            background-color: var(--primary-color);
            border-color: var(--primary-color);
        }
        
        .btn-primary:hover {
            background-color: var(--secondary-color);
            border-color: var(--secondary-color);
        }
        
        .card-header {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center min-vh-100 align-items-center py-5">
            <div class="col-md-5 col-lg-4">
                <div class="card login-card">
                    <div class="card-header text-white text-center py-4">
                        <?php if ($school_branding): ?>
                            <img src="<?= htmlspecialchars($school_logo) ?>" alt="School Logo" class="school-logo mb-3">
                            <h3 class="mb-1"><?= htmlspecialchars($school_branding['name']) ?></h3>
                            <p class="mb-0"><small>Student Portal</small></p>
                        <?php else: ?>
                            <i class="bi bi-person-circle" style="font-size: 3rem;"></i>
                            <h3 class="mb-1">Student Login</h3>
                            <p class="mb-0"><small>Multi-Tenant School System</small></p>
                        <?php endif; ?>
                    </div>
                    
                    <div class="card-body p-4">
                        <?php if ($error): ?>
                            <div class="alert alert-danger alert-dismissible fade show">
                                <i class="bi bi-exclamation-triangle me-1"></i>
                                <?= htmlspecialchars($error) ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($success): ?>
                            <div class="alert alert-success alert-dismissible fade show">
                                <i class="bi bi-check-circle me-1"></i>
                                <?= htmlspecialchars($success) ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>
                        
                        <form method="POST" action="">
                            <input type="hidden" name="csrf_token" value="<?= generateCSRFToken() ?>">
                            
                            <div class="mb-3">
                                <label for="username" class="form-label">
                                    <i class="bi bi-person me-1"></i>Username
                                </label>
                                <input type="text" class="form-control" id="username" name="username" 
                                       value="<?= htmlspecialchars($_POST['username'] ?? '') ?>" 
                                       placeholder="Enter your username" required autofocus>
                            </div>
                            
                            <div class="mb-3">
                                <label for="password" class="form-label">
                                    <i class="bi bi-lock me-1"></i>Password
                                </label>
                                <input type="password" class="form-control" id="password" name="password" 
                                       placeholder="Enter your password" required>
                            </div>
                            
                            <div class="d-grid mb-3">
                                <button type="submit" class="btn btn-primary btn-lg">
                                    <i class="bi bi-box-arrow-in-right me-2"></i>Login
                                </button>
                            </div>
                        </form>
                        
                        <hr>
                        
                        <div class="text-center">
                            <small class="text-muted">
                                <i class="bi bi-info-circle me-1"></i>
                                Don't have an account? 
                                <?php if ($school_branding): ?>
                                    <a href="<?= htmlspecialchars(strtolower($school_branding['code'])) ?>/register">Register here</a>
                                <?php else: ?>
                                    Contact your school administration.
                                <?php endif; ?>
                            </small>
                        </div>
                        
                        <?php if ($school_branding && !empty($school_branding['contact_email'])): ?>
                            <div class="text-center mt-2">
                                <small class="text-muted">
                                    <i class="bi bi-envelope me-1"></i>
                                    Need help? Contact: 
                                    <a href="mailto:<?= htmlspecialchars($school_branding['contact_email']) ?>">
                                        <?= htmlspecialchars($school_branding['contact_email']) ?>
                                    </a>
                                </small>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="card-footer text-center bg-light">
                        <small class="text-muted">
                            <i class="bi bi-shield-lock me-1"></i>
                            Secure Login • Powered by MelaneGroup
                        </small>
                    </div>
                </div>
                
                <!-- Additional Links -->
                <div class="text-center mt-3">
                    <a href="index.php" class="text-white text-decoration-none">
                        <i class="bi bi-house me-1"></i>Back to Home
                    </a>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

