<?php
/**
 * Student Subjects Management
 * View and manage enrolled subjects
 */

session_start();
require_once '../includes/functions.php';
requireStudentLogin();

$student = getCurrentStudent();
$pdo = getDB();

// Get student details
$stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
$stmt->execute([$student['id']]);
$student_details = $stmt->fetch();

// Get school details
$stmt = $pdo->prepare("
    SELECT ar.academy_name, ts.setting_key, ts.setting_value
    FROM academy_references ar
    LEFT JOIN tenant_settings ts ON ar.reference_code = ts.academy_reference
    WHERE ar.reference_code = ?
");
$stmt->execute([$student['academy_reference']]);
$results = $stmt->fetchAll();
$school = ['academy_name' => $results[0]['academy_name'] ?? 'School'];
$settings = [];
foreach ($results as $row) {
    if ($row['setting_key']) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
}

$moodle_url = $settings['moodle_url'] ?? null;

// Get enrolled subjects with details
$stmt = $pdo->prepare("
    SELECT 
        s.id,
        s.name,
        s.code,
        s.description
    FROM student_subject ss
    INNER JOIN subjects s ON ss.subject_id = s.id
    WHERE ss.student_id = ?
    ORDER BY s.name
");
$stmt->execute([$student['id']]);
$enrolled_subjects = $stmt->fetchAll();

// Get available subjects (not enrolled)
$enrolled_ids = array_column($enrolled_subjects, 'id');
$not_enrolled_placeholder = !empty($enrolled_ids) ? str_repeat('?,', count($enrolled_ids) - 1) . '?' : '0';

$sql = "
    SELECT s.*
    FROM subjects s
    INNER JOIN grade_subject gs ON s.id = gs.subject_id
    WHERE gs.grade_id = ?
";

if (!empty($enrolled_ids)) {
    $sql .= " AND s.id NOT IN ($not_enrolled_placeholder)";
    $params = array_merge([$student_details['grade_id']], $enrolled_ids);
} else {
    $params = [$student_details['grade_id']];
}

$sql .= " ORDER BY s.name";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$available_subjects = $stmt->fetchAll();

$pageTitle = "My Subjects - Student Portal";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $pageTitle ?></title>
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    
    <style>
        .sidebar {
            min-height: 100vh;
            background: linear-gradient(135deg, #0d6efd 0%, #0a58ca 100%);
        }
        .nav-link {
            color: rgba(255,255,255,0.8);
            transition: all 0.3s;
        }
        .nav-link:hover, .nav-link.active {
            color: white;
            background: rgba(255,255,255,0.1);
        }
        .subject-card {
            transition: all 0.3s;
            border-left: 4px solid #0d6efd;
        }
        .subject-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 px-0 sidebar text-white">
                <div class="p-3">
                    <h5 class="mb-0">
                        <i class="bi bi-mortarboard-fill me-2"></i>
                        Student Portal
                    </h5>
                    <small class="text-white-50"><?= htmlspecialchars($school['academy_name']) ?></small>
                </div>
                
                <nav class="nav flex-column px-2">
                    <a class="nav-link rounded" href="dashboard.php">
                        <i class="bi bi-house me-2"></i>Dashboard
                    </a>
                    <a class="nav-link active rounded" href="subjects.php">
                        <i class="bi bi-book me-2"></i>My Subjects
                    </a>
                    <a class="nav-link rounded" href="make_payment.php">
                        <i class="bi bi-credit-card me-2"></i>Make Payment
                    </a>
                    <a class="nav-link rounded" href="payment_history.php">
                        <i class="bi bi-receipt me-2"></i>Payment History
                    </a>
                    <?php if ($moodle_url): ?>
                    <a class="nav-link rounded" href="<?= htmlspecialchars($moodle_url) ?>" target="_blank">
                        <i class="bi bi-mortarboard me-2"></i>Moodle Portal
                        <i class="bi bi-box-arrow-up-right ms-1 small"></i>
                    </a>
                    <?php endif; ?>
                    <a class="nav-link rounded" href="profile.php">
                        <i class="bi bi-person me-2"></i>My Profile
                    </a>
                    <hr class="text-white-50">
                    <a class="nav-link rounded" href="logout.php">
                        <i class="bi bi-box-arrow-right me-2"></i>Logout
                    </a>
                </nav>
            </div>
            
            <!-- Main Content -->
            <div class="col-md-9 col-lg-10 px-4 py-3">
                <h2 class="mb-4"><i class="bi bi-book me-2"></i>My Subjects</h2>

                <!-- Enrolled Subjects -->
                <div class="card mb-4">
                    <div class="card-header bg-success text-white">
                        <div class="d-flex justify-content-between align-items-center">
                            <h5 class="mb-0"><i class="bi bi-check-circle me-2"></i>Enrolled Subjects (<?= count($enrolled_subjects) ?>)</h5>
                            <a href="make_payment.php" class="btn btn-light btn-sm">
                                <i class="bi bi-plus-circle me-1"></i>Add More Subjects
                            </a>
                        </div>
                    </div>
                    <div class="card-body">
                        <?php if (empty($enrolled_subjects)): ?>
                            <div class="alert alert-warning">
                                <i class="bi bi-exclamation-triangle me-2"></i>
                                You are not enrolled in any subjects yet.
                                <a href="make_payment.php" class="alert-link">Make a payment to enroll.</a>
                            </div>
                        <?php else: ?>
                            <div class="row">
                                <?php foreach ($enrolled_subjects as $subject): ?>
                                <div class="col-md-6 mb-3">
                                    <div class="card subject-card">
                                        <div class="card-body">
                                            <h6 class="card-title">
                                                <i class="bi bi-book-fill text-success me-2"></i>
                                                <?= htmlspecialchars($subject['name']) ?>
                                            </h6>
                                            <p class="mb-2">
                                                <small class="text-muted">Code: <code><?= htmlspecialchars($subject['code']) ?></code></small>
                                            </p>
                                            <span class="badge bg-success">
                                                <i class="bi bi-check-circle"></i> Active
                                            </span>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Available Subjects -->
                <?php if (!empty($available_subjects)): ?>
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h5 class="mb-0"><i class="bi bi-plus-circle me-2"></i>Available Subjects (<?= count($available_subjects) ?>)</h5>
                    </div>
                    <div class="card-body">
                        <p class="text-muted">
                            <i class="bi bi-info-circle me-1"></i>
                            These subjects are available for your grade. Make a payment to enroll.
                        </p>
                        <div class="row">
                            <?php foreach ($available_subjects as $subject): ?>
                            <div class="col-md-6 mb-3">
                                <div class="card border">
                                    <div class="card-body">
                                        <h6 class="card-title">
                                            <i class="bi bi-book text-primary me-2"></i>
                                            <?= htmlspecialchars($subject['name']) ?>
                                        </h6>
                                        <p class="mb-0">
                                            <small class="text-muted">Code: <code><?= htmlspecialchars($subject['code']) ?></code></small>
                                        </p>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <div class="mt-3">
                            <a href="make_payment.php" class="btn btn-primary">
                                <i class="bi bi-credit-card me-2"></i>Enroll in More Subjects
                            </a>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

