<?php
require_once __DIR__ . '/includes/functions.php';
require_once __DIR__ . '/includes/school_registration_helper.php';
require_once __DIR__ . '/includes/favicon_helper.php';

$errors = [];
$successData = null;
$csrfToken = generateCSRFToken();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = 'Security verification failed. Please refresh the page and try again.';
    } else {
        $schoolName = trim((string)($_POST['school_name'] ?? ''));
        $referenceCode = strtoupper(trim((string)($_POST['reference_code'] ?? '')));
        $adminEmail = trim((string)($_POST['admin_email'] ?? ''));
        $contactName = trim((string)($_POST['contact_name'] ?? ''));
        $contactPhone = preg_replace('/[^0-9+]/', '', (string)($_POST['contact_phone'] ?? ''));
        $notes = trim((string)($_POST['notes'] ?? ''));

        if ($contactPhone !== '' && !preg_match('/^\+?[0-9]{7,15}$/', $contactPhone)) {
            $errors[] = 'Please enter a valid contact phone number.';
        }

        if (empty($errors)) {
            try {
                $generatedPassword = bin2hex(random_bytes(4)) . random_int(10, 99);

                $successData = createSchoolAccount(
                    [
                        'school_name'    => $schoolName,
                        'reference_code' => $referenceCode,
                        'admin_email'    => $adminEmail,
                        'admin_password' => $generatedPassword,
                        'description'    => $notes,
                        'contact_phone'  => $contactPhone,
                        'contact_name'   => $contactName,
                    ],
                    [
                        'created_by'         => null,
                        'send_welcome_email' => true,
                    ]
                );
            } catch (Throwable $e) {
                $errors[] = $e->getMessage();
            }
        }
    }
}
?>
<!doctype html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Register Your School - <?php echo htmlspecialchars(APP_NAME); ?></title>
    <?php echoFavicon(null); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(120deg, rgba(79, 70, 229, 0.08), rgba(16, 185, 129, 0.08));
            min-height: 100vh;
        }
        .card {
            border: none;
            box-shadow: 0 1.5rem 3rem rgba(15, 23, 42, 0.15);
        }
        .step-badge {
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            font-weight: 600;
            background: rgba(79, 70, 229, 0.12);
            color: #4F46E5;
        }
        .form-control, .form-select {
            border-radius: 0.75rem;
            border: 1.5px solid rgba(148, 163, 184, 0.35);
        }
        .form-control:focus, .form-select:focus {
            border-color: #4F46E5;
            box-shadow: 0 0 0 0.15rem rgba(79, 70, 229, 0.15);
        }
        .hero {
            background: radial-gradient(circle at top left, rgba(79, 70, 229, 0.14), transparent 50%);
            border-radius: 1.5rem;
        }
    </style>
</head>
<body>
<div class="container py-5">
    <div class="row justify-content-center">
        <div class="col-lg-10 col-xl-8">
            <div class="hero p-4 p-md-5 mb-4 bg-white shadow-sm">
                <div class="d-flex align-items-center">
                    <span class="badge bg-primary-subtle text-primary-emphasis me-2">Multi-Tenant Platform</span>
                    <small class="text-muted">Empower your school in minutes</small>
                </div>
                <h1 class="display-6 fw-bold mt-3 mb-2">School Registration Portal</h1>
                <p class="lead text-secondary mb-0">
                    Create your school admin account, configure branding, and start enrolling students instantly.
                </p>
            </div>

            <div class="card">
                <div class="card-body p-4 p-md-5">
                    <?php if ($successData): ?>
                        <div class="alert alert-success rounded-4">
                            <div class="d-flex align-items-start">
                                <div class="me-3">
                                    <i class="bi bi-check-circle-fill fs-2 text-success"></i>
                                </div>
                                <div>
                                    <h4 class="alert-heading">Your school account is ready!</h4>
                                    <p class="mb-2">
                                        We have emailed the administrator credentials to <strong><?php echo htmlspecialchars($successData['admin_email']); ?></strong>.
                                        Follow the link below to access your dashboard.
                                    </p>
                                    <p class="mb-3">
                                        <span class="fw-semibold">Admin Username:</span>
                                        <code><?php echo htmlspecialchars($successData['admin_username']); ?></code><br>
                                        <span class="fw-semibold">School Code:</span>
                                        <code><?php echo htmlspecialchars($successData['reference_code']); ?></code>
                                    </p>
                                    <a class="btn btn-success rounded-pill" href="<?php echo htmlspecialchars($successData['admin_login_url']); ?>">
                                        <i class="bi bi-box-arrow-in-right me-1"></i>Go to Admin Login
                                    </a>
                                    <a class="btn btn-outline-primary rounded-pill ms-2" href="index.php">
                                        Back to Home
                                    </a>
                                </div>
                            </div>
                        </div>
                    <?php else: ?>
                        <?php if (!empty($errors)): ?>
                            <div class="alert alert-danger rounded-4">
                                <h5 class="alert-heading"><i class="bi bi-exclamation-triangle-fill me-2"></i>We need your attention</h5>
                                <ul class="mb-0 ps-3">
                                    <?php foreach ($errors as $error): ?>
                                        <li><?php echo htmlspecialchars($error); ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        <?php endif; ?>

                        <form method="post" novalidate>
                            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrfToken); ?>">
                            
                            <div class="mb-4">
                                <div class="d-flex align-items-center mb-2">
                                    <span class="step-badge me-2">1</span>
                                    <div>
                                        <h5 class="mb-0">School Details</h5>
                                        <small class="text-muted">Basic information used across your portal</small>
                                    </div>
                                </div>
                                <div class="row g-3 mt-1">
                                    <div class="col-md-8">
                                        <label class="form-label fw-semibold">Official School Name</label>
                                        <input type="text" class="form-control form-control-lg" name="school_name"
                                               value="<?php echo htmlspecialchars($_POST['school_name'] ?? ''); ?>"
                                               placeholder="e.g. Royal High Academy" required>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label fw-semibold">Preferred School Code</label>
                                        <input type="text" class="form-control form-control-lg text-uppercase"
                                               name="reference_code" maxlength="10"
                                               value="<?php echo htmlspecialchars($_POST['reference_code'] ?? ''); ?>"
                                               placeholder="e.g. ROYAL" required>
                                        <small class="text-muted">2-10 letters or numbers</small>
                                    </div>
                                </div>
                            </div>

                            <div class="mb-4">
                                <div class="d-flex align-items-center mb-2">
                                    <span class="step-badge me-2">2</span>
                                    <div>
                                        <h5 class="mb-0">Administrator Contact</h5>
                                        <small class="text-muted">The primary contact who manages the school</small>
                                    </div>
                                </div>
                                <div class="row g-3 mt-1">
                                    <div class="col-md-6">
                                        <label class="form-label fw-semibold">Administrator Full Name</label>
                                        <input type="text" class="form-control form-control-lg" name="contact_name"
                                               value="<?php echo htmlspecialchars($_POST['contact_name'] ?? ''); ?>"
                                               placeholder="e.g. Jane Doe" required>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label fw-semibold">Administrator Email</label>
                                        <input type="email" class="form-control form-control-lg" name="admin_email"
                                               value="<?php echo htmlspecialchars($_POST['admin_email'] ?? ''); ?>"
                                               placeholder="e.g. admin@royalhigh.ac.sz" required>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label fw-semibold">Contact Phone (Optional)</label>
                                        <input type="text" class="form-control form-control-lg" name="contact_phone"
                                               value="<?php echo htmlspecialchars($_POST['contact_phone'] ?? ''); ?>"
                                               placeholder="e.g. +26876123456">
                                    </div>
                                </div>
                            </div>

                            <div class="mb-4">
                                <div class="d-flex align-items-center mb-2">
                                    <span class="step-badge me-2">3</span>
                                    <div>
                                        <h5 class="mb-0">Additional Information</h5>
                                        <small class="text-muted">Help us set up the right defaults for you</small>
                                    </div>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label fw-semibold">Notes (Optional)</label>
                                    <textarea class="form-control" rows="3" name="notes"
                                              placeholder="Share any goals, current enrolment, or special requirements"><?php
                                            echo htmlspecialchars($_POST['notes'] ?? '');
                                        ?></textarea>
                                </div>
                            </div>

                            <div class="alert alert-info rounded-4">
                                <h6 class="fw-semibold mb-2"><i class="bi bi-info-circle me-2"></i>Before you submit</h6>
                                <ul class="mb-0 ps-3">
                                    <li>Your admin credentials will be emailed instantly.</li>
                                    <li>Use the provided school code when sharing registration links.</li>
                                    <li>You can customise branding, fees, and grades after login.</li>
                                </ul>
                            </div>

                            <div class="d-grid d-md-flex justify-content-md-end gap-2">
                                <a class="btn btn-outline-secondary rounded-pill" href="index.php">
                                    <i class="bi bi-arrow-left me-1"></i>Back to Home
                                </a>
                                <button type="submit" class="btn btn-primary rounded-pill px-4">
                                    <i class="bi bi-building-add me-2"></i>Register School
                                </button>
                            </div>
                        </form>
                    <?php endif; ?>
                </div>
            </div>

            <div class="text-center text-muted mt-4">
                <small>&copy; <?php echo date('Y'); ?> <?php echo htmlspecialchars(APP_NAME); ?>. All rights reserved.</small>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

