<?php
/**
 * Card Payment Notification Handler (PayGate)
 * Handles server-to-server notifications from PayGate
 */

require_once __DIR__ . '/../../config.php';
require_once __DIR__ . '/../../includes/functions.php';
require_once __DIR__ . '/../../includes/wallet_functions.php';

// Log notification for debugging
function logNotify($message) {
    $logFile = __DIR__ . '/../../logs/paygate_notify.log';
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] $message\n", FILE_APPEND | LOCK_EX);
}

// Get notification data
$input = file_get_contents('php://input');
logNotify("Notification received: " . $input);

// Parse POST data
parse_str($input, $notify_data);

// Verify required fields
$pay_request_id = $notify_data['PAY_REQUEST_ID'] ?? null;
$reference = $notify_data['REFERENCE'] ?? null;
$transaction_status = $notify_data['TRANSACTION_STATUS'] ?? null;

if (!$pay_request_id || !$reference) {
    logNotify("Missing required fields");
    http_response_code(400);
    echo "ERROR: Missing required fields";
    exit;
}

// Verify checksum
$verify_checksum = '';
foreach ($notify_data as $key => $value) {
    if ($key !== 'CHECKSUM') {
        $verify_checksum .= $value;
    }
}
$verify_checksum .= PAYGATE_SECRET;

if (md5($verify_checksum) !== ($notify_data['CHECKSUM'] ?? '')) {
    logNotify("Invalid checksum");
    http_response_code(400);
    echo "ERROR: Invalid checksum";
    exit;
}

logNotify("Checksum verified. Processing payment: " . $reference);

try {
    $pdo = getDB();
    
    // Find payment
    $stmt = $pdo->prepare("SELECT * FROM payments WHERE reference_id = ?");
    $stmt->execute([$reference]);
    $payment = $stmt->fetch();
    
    if (!$payment) {
        logNotify("Payment not found: " . $reference);
        http_response_code(404);
        echo "ERROR: Payment not found";
        exit;
    }
    
    logNotify("Found payment ID: " . $payment['id']);
    
    // Process based on status
    if ($transaction_status == '1') {
        // Payment approved
        $stmt = $pdo->prepare("
            UPDATE payments 
            SET status = 'SUCCESSFUL',
                momo_status = 'APPROVED',
                transaction_id = ?,
                card_type = ?,
                card_last4 = ?,
                raw_response = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([
            $notify_data['TRANSACTION_ID'] ?? null,
            $notify_data['PAY_METHOD'] ?? null,
            substr($notify_data['PAY_METHOD_DETAIL'] ?? '', -4),
            json_encode($notify_data),
            $payment['id']
        ]);
        
        // Update student
        $stmt = $pdo->prepare("
            UPDATE students 
            SET payment_status = 'PAID',
                payment_date = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$payment['student_id']]);
        
        // Credit admin wallet
        $academy_reference = $payment['academy_reference'] ?? 'KINE';
        $wallet_result = creditAdminWallet($payment['id'], PLATFORM_ADMIN_ID, $academy_reference);
        
        if ($wallet_result['success']) {
            logNotify("Admin wallet credited successfully");
        } else {
            logNotify("Failed to credit wallet: " . $wallet_result['error']);
        }
        
        logNotify("Payment approved and processed");
        
    } else {
        // Payment declined or cancelled
        $status_desc = $transaction_status == '4' ? 'Cancelled' : 'Declined';
        
        $stmt = $pdo->prepare("
            UPDATE payments 
            SET status = 'FAILED',
                momo_status = ?,
                error_message = ?,
                raw_response = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([
            $status_desc,
            $notify_data['RESULT_DESC'] ?? 'Payment ' . strtolower($status_desc),
            json_encode($notify_data),
            $payment['id']
        ]);
        
        logNotify("Payment " . strtolower($status_desc));
    }
    
    // Respond to PayGate
    http_response_code(200);
    echo "OK";
    logNotify("Notification processed successfully");
    
} catch (Exception $e) {
    logNotify("Error: " . $e->getMessage());
    http_response_code(500);
    echo "ERROR: " . $e->getMessage();
}
?>

