<?php
/**
 * Payment Page for Reminder Links
 * Handles payment from reminder/warning emails
 */

session_start();
require_once 'config.php';
require_once 'includes/functions.php';
require_once 'includes/payment_reminder_manager.php';
require_once 'includes/url_helper.php';
require_once 'includes/favicon_helper.php';

$pdo = getDB();
$error = '';
$student = null;
$amount = 0;
$token_data = null;

// Get and validate token
$token = $_GET['token'] ?? '';

if (empty($token)) {
    $error = "Invalid payment link. No token provided.";
} else {
    // Decrypt and validate token
    $manager = new PaymentReminderManager();
    $token_result = $manager->decryptToken($token);
    
    if (!$token_result['success']) {
        $error = $token_result['error'];
    } else {
        $token_data = $token_result['data'];
        $student_id = $token_data['student_id'];
        $amount = $token_data['amount'];
        
        // Get student details
        $stmt = $pdo->prepare("
            SELECT s.*, sps.*, ar.academy_name
            FROM students s
            INNER JOIN student_payment_schedules sps ON s.id = sps.student_id
            INNER JOIN academy_references ar ON s.academy_reference = ar.reference_code
            WHERE s.id = ?
        ");
        $stmt->execute([$student_id]);
        $student = $stmt->fetch();
        
        if (!$student) {
            $error = "Student not found.";
        } else {
            // Update reminder click tracking
            $pdo->prepare("
                UPDATE payment_reminders 
                SET link_clicked = 1, link_clicked_at = NOW()
                WHERE payment_token = ?
            ")->execute([$token]);
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Make Payment - <?= APP_NAME ?></title>
    <?php echoFavicon($student['academy_reference'] ?? null); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px 0;
        }
        .payment-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        .payment-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        .amount-display {
            font-size: 48px;
            font-weight: bold;
            color: #667eea;
            margin: 20px 0;
        }
        .payment-method {
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            padding: 20px;
            margin: 10px 0;
            cursor: pointer;
            transition: all 0.3s;
        }
        .payment-method:hover {
            border-color: #667eea;
            background: #f8f9ff;
        }
        .payment-method.selected {
            border-color: #667eea;
            background: #e7f1ff;
        }
        .btn-pay {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border: none;
            color: white;
            padding: 15px 40px;
            font-size: 18px;
            font-weight: bold;
            border-radius: 10px;
        }
        .btn-pay:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(102, 126, 234, 0.4);
        }
        .info-box {
            background: #f8f9fa;
            border-left: 4px solid #667eea;
            padding: 15px;
            margin: 20px 0;
        }
        .status-badge {
            display: inline-block;
            padding: 5px 15px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: bold;
        }
        .status-overdue {
            background: #dc3545;
            color: white;
        }
        .status-pending {
            background: #ffc107;
            color: #000;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-8 col-lg-6">
                
                <?php if ($error): ?>
                <!-- Error State -->
                <div class="payment-card">
                    <div class="payment-header">
                        <i class="bi bi-exclamation-triangle" style="font-size: 60px;"></i>
                        <h2 class="mt-3">Payment Link Invalid</h2>
                    </div>
                    <div class="p-4 text-center">
                        <div class="alert alert-danger">
                            <?= htmlspecialchars($error) ?>
                        </div>
                        <p>This payment link may have expired or is invalid. Please contact your school administration for assistance.</p>
                        <a href="<?= generateURL('index.php') ?>" class="btn btn-primary mt-3">
                            <i class="bi bi-house me-2"></i>Go to Homepage
                        </a>
                    </div>
                </div>
                
                <?php elseif ($student): ?>
                <!-- Payment Form -->
                <div class="payment-card">
                    <div class="payment-header">
                        <i class="bi bi-credit-card" style="font-size: 60px;"></i>
                        <h2 class="mt-3">Make Payment</h2>
                        <p class="mb-0"><?= htmlspecialchars($student['academy_name']) ?></p>
                    </div>
                    
                    <div class="p-4">
                        <!-- Student Info -->
                        <div class="text-center mb-4">
                            <h4><?= htmlspecialchars($student['full_name']) ?></h4>
                            <p class="text-muted mb-1"><?= htmlspecialchars($student['email']) ?></p>
                            <span class="status-badge status-<?= $student['payment_status'] ?>">
                                <?= strtoupper($student['payment_status']) ?>
                            </span>
                        </div>
                        
                        <!-- Amount Due -->
                        <div class="text-center">
                            <p class="text-muted mb-1">Amount Due:</p>
                            <div class="amount-display">
                                SZL <?= number_format($amount, 2) ?>
                            </div>
                        </div>
                        
                        <!-- Payment Details -->
                        <div class="info-box">
                            <h6><i class="bi bi-info-circle me-2"></i>Payment Information</h6>
                            <div class="row mt-3">
                                <div class="col-6">
                                    <small class="text-muted">Last Payment:</small><br>
                                    <strong><?= $student['last_payment_date'] ? date('M j, Y', strtotime($student['last_payment_date'])) : 'Never' ?></strong>
                                </div>
                                <div class="col-6">
                                    <small class="text-muted">Due Date:</small><br>
                                    <strong><?= date('M j, Y', strtotime($student['next_due_date'])) ?></strong>
                                </div>
                            </div>
                            <?php if ($student['payment_status'] === 'overdue'): ?>
                            <div class="alert alert-warning mt-3 mb-0">
                                <small><i class="bi bi-exclamation-triangle me-1"></i>
                                <strong>Payment Overdue!</strong> Please pay immediately to avoid account suspension.</small>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Payment Method Selection -->
                        <h5 class="mb-3">Select Payment Method:</h5>
                        
                        <form method="POST" action="process_reminder_payment.php" id="paymentForm">
                            <input type="hidden" name="token" value="<?= htmlspecialchars($token) ?>">
                            <input type="hidden" name="student_id" value="<?= $student['id'] ?>">
                            <input type="hidden" name="amount" value="<?= $amount ?>">
                            <input type="hidden" name="schedule_id" value="<?= $student['id'] ?>">
                            <input type="hidden" name="academy_reference" value="<?= htmlspecialchars($student['academy_reference']) ?>">
                            <input type="hidden" name="payment_method" id="selected_method" value="">
                            
                            <!-- MTN Mobile Money -->
                            <div class="payment-method" onclick="selectMethod('momo')">
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="payment_method_radio" id="method_momo" value="momo">
                                    <label class="form-check-label w-100" for="method_momo">
                                        <div class="d-flex align-items-center">
                                            <i class="bi bi-phone" style="font-size: 32px; color: #FFCB05; margin-right: 15px;"></i>
                                            <div>
                                                <h6 class="mb-0">MTN Mobile Money</h6>
                                                <small class="text-muted">Pay with your MoMo wallet</small>
                                            </div>
                                        </div>
                                    </label>
                                </div>
                            </div>
                            
                            <!-- Card Payment -->
                            <div class="payment-method" onclick="selectMethod('card')">
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="payment_method_radio" id="method_card" value="card">
                                    <label class="form-check-label w-100" for="method_card">
                                        <div class="d-flex align-items-center">
                                            <i class="bi bi-credit-card" style="font-size: 32px; color: #667eea; margin-right: 15px;"></i>
                                            <div>
                                                <h6 class="mb-0">Credit/Debit Card</h6>
                                                <small class="text-muted">Visa, Mastercard, etc.</small>
                                            </div>
                                        </div>
                                    </label>
                                </div>
                            </div>
                            
                            <!-- MoMo Phone Number (shown when MoMo selected) -->
                            <div id="momo_phone_section" style="display: none;" class="mt-3">
                                <label class="form-label">MTN Mobile Money Number:</label>
                                <div class="input-group">
                                    <span class="input-group-text">+268</span>
                                    <input type="text" class="form-control" name="momo_phone" id="momo_phone" 
                                           placeholder="7XXXXXXX" pattern="[0-9]{8}" maxlength="8"
                                           value="<?= htmlspecialchars(ltrim($student['phone'] ?? '', '+268')) ?>">
                                </div>
                                <small class="text-muted">Enter your 8-digit MTN number</small>
                            </div>
                            
                            <!-- Submit Button -->
                            <div class="d-grid gap-2 mt-4">
                                <button type="submit" class="btn btn-pay" id="submitBtn" disabled>
                                    <i class="bi bi-lock-fill me-2"></i>Proceed to Payment
                                </button>
                            </div>
                        </form>
                        
                        <!-- Security Notice -->
                        <div class="text-center mt-4">
                            <small class="text-muted">
                                <i class="bi bi-shield-check me-1"></i>
                                Secure payment processing
                            </small>
                        </div>
                    </div>
                </div>
                
                <?php endif; ?>
                
                <!-- Footer -->
                <div class="text-center mt-3">
                    <p class="text-white">
                        <small>Need help? Contact school administration</small>
                    </p>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function selectMethod(method) {
            // Update hidden input
            document.getElementById('selected_method').value = method;
            
            // Update radio button
            document.getElementById('method_' + method).checked = true;
            
            // Update visual selection
            document.querySelectorAll('.payment-method').forEach(el => {
                el.classList.remove('selected');
            });
            event.currentTarget.classList.add('selected');
            
            // Show/hide MoMo phone input
            if (method === 'momo') {
                document.getElementById('momo_phone_section').style.display = 'block';
                document.getElementById('momo_phone').required = true;
            } else {
                document.getElementById('momo_phone_section').style.display = 'none';
                document.getElementById('momo_phone').required = false;
            }
            
            // Enable submit button
            document.getElementById('submitBtn').disabled = false;
        }
        
        // Form validation
        document.getElementById('paymentForm').addEventListener('submit', function(e) {
            const method = document.getElementById('selected_method').value;
            
            if (!method) {
                e.preventDefault();
                alert('Please select a payment method');
                return false;
            }
            
            if (method === 'momo') {
                const phone = document.getElementById('momo_phone').value;
                if (!phone || phone.length !== 8) {
                    e.preventDefault();
                    alert('Please enter a valid 8-digit MTN number');
                    return false;
                }
            }
            
            // Disable button to prevent double submission
            document.getElementById('submitBtn').disabled = true;
            document.getElementById('submitBtn').innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Processing...';
        });
    </script>
</body>
</html>

