<?php
/**
 * SMS Functions for Multi-Tenant Platform
 * Supports: Africa's Talking (primary), Twilio (backup)
 */

require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/functions.php';

use AfricasTalking\SDK\AfricasTalking;
use Twilio\Rest\Client as TwilioClient;

/**
 * Send verification code via SMS
 */
function sendVerificationSMS($phone, $student_name, $verification_code, $school_name = null) {
    try {
        $school_name = $school_name ?? 'Your School';
        $message = "Hi {$student_name}, your verification code is: {$verification_code}\n\nValid for " 
                   . SMS_VERIFICATION_EXPIRY . " minutes.\n- {$school_name}";
        
        $result = sendSMS($phone, $message, 'VERIFICATION');
        
        if ($result['success']) {
            // Log verification code
            logSMSVerification($phone, $verification_code);
        }
        
        return $result;
    } catch (Exception $e) {
        error_log("SMS verification failed: " . $e->getMessage());
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Send login credentials via SMS
 */
function sendCredentialsSMS($phone, $student_name, $username, $password, $login_url, $school_name = null) {
    try {
        $school_name = $school_name ?? 'Your School';
        $message = "Welcome {$student_name}!\n\n"
                 . "Username: {$username}\n"
                 . "Password: {$password}\n\n"
                 . "Login: {$login_url}\n\n"
                 . "Keep this safe!\n- {$school_name}";
        
        return sendSMS($phone, $message, 'CREDENTIALS');
    } catch (Exception $e) {
        error_log("SMS credentials failed: " . $e->getMessage());
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Send payment reminder via SMS
 */
function sendPaymentReminderSMS($phone, $student_name, $amount, $due_date, $payment_url) {
    try {
        $message = "Payment Reminder\n\n"
                 . "{$student_name}, your fee of E{$amount} is due on {$due_date}.\n\n"
                 . "Pay now: {$payment_url}";
        
        return sendSMS($phone, $message, 'PAYMENT_REMINDER');
    } catch (Exception $e) {
        error_log("SMS payment reminder failed: " . $e->getMessage());
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Send payment confirmation via SMS
 */
function sendPaymentConfirmationSMS($phone, $student_name, $amount, $receipt_no) {
    try {
        $message = "Payment Received ✓\n\n"
                 . "Student: {$student_name}\n"
                 . "Amount: E{$amount}\n"
                 . "Receipt: {$receipt_no}\n\n"
                 . "Thank you!";
        
        return sendSMS($phone, $message, 'PAYMENT_CONFIRMATION');
    } catch (Exception $e) {
        error_log("SMS payment confirmation failed: " . $e->getMessage());
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Get SMS configuration (tenant-specific or super admin default)
 */
function getSMSConfig($academy_reference = null) {
    $pdo = getDB();
    $config = [];
    
    // Try to get current school from session if not provided
    if (!$academy_reference && function_exists('getCurrentSchool')) {
        try {
            $academy_reference = getCurrentSchool();
        } catch (Exception $e) {
            // Continue without academy_reference
        }
    }
    
    // Try tenant-specific settings first
    if ($academy_reference) {
        $stmt = $pdo->prepare("
            SELECT setting_key, setting_value 
            FROM tenant_settings 
            WHERE academy_reference = ? 
            AND setting_key IN ('use_custom_sms', 'sms_enabled', 'sms_provider', 'sms_username', 
                                'sms_api_key', 'sms_sender_id', 'sms_twilio_sid', 'sms_twilio_token', 'sms_twilio_from')
        ");
        $stmt->execute([$academy_reference]);
        while ($row = $stmt->fetch()) {
            $config[$row['setting_key']] = $row['setting_value'];
        }
        
        // If tenant has custom SMS enabled and configured, use it
        if (($config['use_custom_sms'] ?? 0) && ($config['sms_enabled'] ?? 0)) {
            return $config;
        }
    }
    
    // Fall back to super admin default settings
    $stmt = $pdo->query("
        SELECT setting_key, setting_value 
        FROM super_admin_settings 
        WHERE setting_key LIKE 'default_sms_%' OR setting_key LIKE 'default_at_%' OR setting_key LIKE 'default_twilio_%'
    ");
    $defaultConfig = [];
    while ($row = $stmt->fetch()) {
        // Remove 'default_' prefix and map to config keys
        $key = str_replace('default_', '', $row['setting_key']);
        $key = str_replace('at_', 'sms_', $key); // Map at_ to sms_
        $defaultConfig[$key] = $row['setting_value'];
    }
    
    // Check if default SMS is enabled
    if (($defaultConfig['sms_enabled'] ?? 0) == '1') {
        return $defaultConfig;
    }
    
    // SMS is disabled everywhere
    return ['sms_enabled' => 0];
}

/**
 * Core SMS sending function with provider fallback
 */
function sendSMS($phone, $message, $type = 'GENERAL', $academy_reference = null) {
    // Get SMS configuration (tenant-specific or default)
    $config = getSMSConfig($academy_reference);
    
    // Check if SMS is enabled
    if (!($config['sms_enabled'] ?? 0)) {
        error_log("SMS is disabled for this tenant/globally");
        return ['success' => false, 'error' => 'SMS service is disabled'];
    }
    
    // Format phone number
    $phone = formatPhoneNumber($phone);
    
    if (!$phone) {
        return ['success' => false, 'error' => 'Invalid phone number'];
    }
    
    // Determine provider
    $provider = $config['sms_provider'] ?? 'africastalking';
    
    if ($provider === 'africastalking') {
        $result = sendSMS_AfricasTalking($phone, $message, $config);
    } elseif ($provider === 'twilio') {
        $result = sendSMS_Twilio($phone, $message, $config);
    } else {
        return ['success' => false, 'error' => 'Unknown SMS provider'];
    }
    
    // Log SMS activity
    logSMS($phone, $message, $type, $result['success'] ? 'SUCCESS' : 'FAILED', 
           $result['error'] ?? null, $provider);
    
    return $result;
}

/**
 * Send SMS via Africa's Talking
 */
function sendSMS_AfricasTalking($phone, $message, $config) {
    try {
        // Get credentials from config
        $username = $config['sms_username'] ?? '';
        $apiKey = $config['sms_api_key'] ?? '';
        $senderId = $config['sms_sender_id'] ?? null;
        
        // Check if API key is configured
        if (empty($apiKey) || $apiKey === 'your-api-key-here') {
            error_log("Africa's Talking API key not configured");
            return ['success' => false, 'error' => 'SMS not configured'];
        }
        
        $AT = new AfricasTalking($username, $apiKey);
        $sms = $AT->sms();
        
        $result = $sms->send([
            'to' => $phone,
            'message' => $message,
            'from' => $senderId
        ]);
        
        // Check response
        if (isset($result['data']['SMSMessageData']['Recipients'][0])) {
            $recipient = $result['data']['SMSMessageData']['Recipients'][0];
            
            if ($recipient['status'] === 'Success' || $recipient['statusCode'] == 101) {
                return [
                    'success' => true,
                    'message_id' => $recipient['messageId'] ?? null,
                    'cost' => $recipient['cost'] ?? null,
                    'provider' => 'africastalking'
                ];
            } else {
                return [
                    'success' => false,
                    'error' => $recipient['status'] ?? 'Unknown error',
                    'provider' => 'africastalking'
                ];
            }
        }
        
        return ['success' => false, 'error' => 'Invalid response from provider'];
        
    } catch (Exception $e) {
        error_log("Africa's Talking SMS Error: " . $e->getMessage());
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Send SMS via Twilio
 */
function sendSMS_Twilio($phone, $message, $config) {
    try {
        // Get credentials from config
        $accountSid = $config['sms_twilio_sid'] ?? '';
        $authToken = $config['sms_twilio_token'] ?? '';
        $fromNumber = $config['sms_twilio_from'] ?? '';
        
        // Check if Twilio is configured
        if (empty($accountSid) || empty($authToken)) {
            return ['success' => false, 'error' => 'Twilio not configured'];
        }
        
        $client = new TwilioClient($accountSid, $authToken);
        
        $result = $client->messages->create($phone, [
            'from' => $fromNumber,
            'body' => $message
        ]);
        
        if ($result->sid) {
            return [
                'success' => true,
                'message_id' => $result->sid,
                'provider' => 'twilio'
            ];
        }
        
        return ['success' => false, 'error' => 'Failed to send SMS'];
        
    } catch (Exception $e) {
        error_log("Twilio SMS Error: " . $e->getMessage());
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Format phone number to international format
 */
function formatPhoneNumber($phone) {
    // Remove spaces, dashes, parentheses
    $phone = preg_replace('/[^0-9+]/', '', $phone);
    
    // If already has +, validate and return
    if (str_starts_with($phone, '+')) {
        return strlen($phone) >= 10 ? $phone : null;
    }
    
    // Remove leading zeros
    $phone = ltrim($phone, '0');
    
    // Add country code if not present
    if (!str_starts_with($phone, '268')) {
        $phone = '268' . $phone;
    }
    
    // Add + prefix
    $phone = '+' . $phone;
    
    // Validate Eswatini number format (+268 followed by 8 digits)
    if (preg_match('/^\+268\d{8}$/', $phone)) {
        return $phone;
    }
    
    error_log("Invalid phone number format: " . $phone);
    return null;
}

/**
 * Generate verification code
 */
function generateVerificationCode($length = 6) {
    return str_pad(random_int(0, pow(10, $length) - 1), $length, '0', STR_PAD_LEFT);
}

/**
 * Log SMS activity
 */
function logSMS($phone, $message, $type, $status, $error = null, $provider = null) {
    try {
        $pdo = getDB();
        $stmt = $pdo->prepare("
            INSERT INTO sms_logs 
            (phone_number, message, message_type, status, error_message, provider, sent_at) 
            VALUES (?, ?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([$phone, $message, $type, $status, $error, $provider]);
    } catch (Exception $e) {
        error_log("Failed to log SMS: " . $e->getMessage());
    }
}

/**
 * Log verification code
 */
function logSMSVerification($phone, $code) {
    try {
        $pdo = getDB();
        $expiry = date('Y-m-d H:i:s', strtotime('+' . SMS_VERIFICATION_EXPIRY . ' minutes'));
        
        $stmt = $pdo->prepare("
            INSERT INTO sms_verifications 
            (phone_number, verification_code, expires_at, created_at) 
            VALUES (?, ?, ?, NOW())
        ");
        $stmt->execute([$phone, $code, $expiry]);
    } catch (Exception $e) {
        error_log("Failed to log verification: " . $e->getMessage());
    }
}

/**
 * Verify SMS code
 */
function verifySMSCode($phone, $code) {
    try {
        $pdo = getDB();
        $phone = formatPhoneNumber($phone);
        
        $stmt = $pdo->prepare("
            SELECT * FROM sms_verifications 
            WHERE phone_number = ? 
            AND verification_code = ? 
            AND expires_at > NOW() 
            AND used = 0
            ORDER BY created_at DESC 
            LIMIT 1
        ");
        $stmt->execute([$phone, $code]);
        $verification = $stmt->fetch();
        
        if ($verification) {
            // Mark as used
            $stmt = $pdo->prepare("
                UPDATE sms_verifications 
                SET used = 1, verified_at = NOW() 
                WHERE id = ?
            ");
            $stmt->execute([$verification['id']]);
            
            return ['success' => true, 'message' => 'Code verified successfully'];
        }
        
        return ['success' => false, 'error' => 'Invalid or expired code'];
        
    } catch (Exception $e) {
        error_log("Verification error: " . $e->getMessage());
        return ['success' => false, 'error' => 'Verification failed'];
    }
}

/**
 * Test SMS configuration
 */
function testSMSConfiguration($test_phone = null) {
    try {
        $test_phone = $test_phone ?? '+268' . '76000000'; // Test number
        $test_message = "Test SMS from " . APP_NAME . ". Your SMS is configured correctly!";
        
        $result = sendSMS($test_phone, $test_message, 'TEST');
        
        return $result;
        
    } catch (Exception $e) {
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Get SMS statistics
 */
function getSMSStatistics() {
    try {
        $pdo = getDB();
        
        // Total SMS sent
        $stmt = $pdo->query("SELECT COUNT(*) as total FROM sms_logs");
        $total = $stmt->fetch()['total'];
        
        // Successful SMS
        $stmt = $pdo->query("SELECT COUNT(*) as success FROM sms_logs WHERE status = 'SUCCESS'");
        $success = $stmt->fetch()['success'];
        
        // Failed SMS
        $stmt = $pdo->query("SELECT COUNT(*) as failed FROM sms_logs WHERE status = 'FAILED'");
        $failed = $stmt->fetch()['failed'];
        
        // Today's SMS
        $stmt = $pdo->query("SELECT COUNT(*) as today FROM sms_logs WHERE DATE(sent_at) = CURDATE()");
        $today = $stmt->fetch()['today'];
        
        return [
            'total' => $total,
            'success' => $success,
            'failed' => $failed,
            'today' => $today,
            'success_rate' => $total > 0 ? round(($success / $total) * 100, 2) : 0
        ];
        
    } catch (Exception $e) {
        error_log("Failed to get SMS statistics: " . $e->getMessage());
        return [
            'total' => 0,
            'success' => 0,
            'failed' => 0,
            'today' => 0,
            'success_rate' => 0
        ];
    }
}

