<?php
/**
 * Example: Upload Student Photo (Updated for Tenant System)
 * 
 * This replaces old direct upload with tenant-aware upload
 */

session_start();
require_once '../config.php';
require_once '../includes/tenant_directory_manager.php';
require_once '../includes/tenant_file_upload.php';

// Check authentication
if (!isset($_SESSION['user_id']) || !isset($_SESSION['academy_reference'])) {
    die(json_encode(['success' => false, 'error' => 'Not authenticated']));
}

// Database connection
$conn = new PDO(
    "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
    DB_USER,
    DB_PASS
);

// Initialize upload handler
$dirManager = new TenantDirectoryManager($conn);
$uploader = new TenantFileUpload($conn, $dirManager);

// Handle upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['photo'])) {
    $student_id = (int)$_POST['student_id'];
    
    // Verify student belongs to this tenant
    $stmt = $conn->prepare("
        SELECT id FROM students 
        WHERE id = ? AND academy_reference = ?
    ");
    $stmt->execute([$student_id, $_SESSION['academy_reference']]);
    
    if (!$stmt->fetch()) {
        die(json_encode(['success' => false, 'error' => 'Student not found']));
    }
    
    // Upload photo using tenant system
    $result = $uploader->uploadProfilePhoto(
        $_SESSION['academy_reference'],
        $student_id,
        $_FILES['photo']
    );
    
    // Return JSON response
    header('Content-Type: application/json');
    echo json_encode($result);
    exit;
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>Upload Student Photo</title>
</head>
<body>
    <h2>Upload Student Photo</h2>
    
    <form method="post" enctype="multipart/form-data" id="upload-form">
        <label>Student ID:</label>
        <input type="number" name="student_id" required><br><br>
        
        <label>Profile Photo:</label>
        <input type="file" name="photo" accept="image/jpeg,image/png" required><br><br>
        
        <button type="submit">Upload Photo</button>
    </form>
    
    <div id="result"></div>
    
    <script>
    document.getElementById('upload-form').addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const formData = new FormData(e.target);
        const response = await fetch('', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        const resultDiv = document.getElementById('result');
        
        if (result.success) {
            resultDiv.innerHTML = `
                <p style="color:green;">✓ Photo uploaded successfully!</p>
                <img src="${result.url}" alt="Photo" style="max-width:200px;">
            `;
        } else {
            resultDiv.innerHTML = `<p style="color:red;">✗ Error: ${result.error}</p>`;
        }
    });
    </script>
</body>
</html>

