-- =====================================================
-- TENANT FILES TRACKING TABLE - MIGRATION
-- =====================================================
-- Purpose: Track all files uploaded by tenants
-- Author: Multi-Tenant System
-- Date: 2025-01-30
-- Version: 1.0
-- =====================================================

-- =====================================================
-- STEP 1: BACKUP EXISTING DATA (IF TABLE EXISTS)
-- =====================================================

-- Create backup table if tenant_files already exists
CREATE TABLE IF NOT EXISTS tenant_files_backup_20250130 AS 
SELECT * FROM tenant_files WHERE 1=0;

-- If table exists, backup data
INSERT INTO tenant_files_backup_20250130 
SELECT * FROM tenant_files WHERE EXISTS (
    SELECT 1 FROM information_schema.TABLES 
    WHERE TABLE_SCHEMA = DATABASE() 
    AND TABLE_NAME = 'tenant_files'
);

-- =====================================================
-- STEP 2: DROP EXISTING TABLES (IF EXISTS)
-- =====================================================
-- Note: Must drop in correct order (child tables first)

-- Drop views first
DROP VIEW IF EXISTS v_active_tenant_files;
DROP VIEW IF EXISTS v_tenant_file_stats;

-- Drop triggers
DROP TRIGGER IF EXISTS before_tenant_file_access_insert;

-- Drop child tables (with foreign keys pointing to tenant_files)
DROP TABLE IF EXISTS tenant_file_versions;
DROP TABLE IF EXISTS tenant_file_access_log;

-- Drop parent table
DROP TABLE IF EXISTS tenant_files;

-- =====================================================
-- STEP 3: CREATE TENANT FILES TABLE
-- =====================================================

CREATE TABLE tenant_files (
    -- Primary Key
    id INT AUTO_INCREMENT PRIMARY KEY,
    
    -- Tenant Information
    tenant_id CHAR(36) NOT NULL COMMENT 'Tenant/School identifier',
    
    -- File Classification
    file_category ENUM(
        'document',      -- General documents (PDFs, Word, Excel)
        'photo',         -- Profile photos, images
        'receipt',       -- Payment receipts
        'report',        -- Generated reports
        'backup',        -- System backups
        'other'          -- Miscellaneous files
    ) NOT NULL DEFAULT 'other' COMMENT 'File category for organization',
    
    -- File Information
    original_filename VARCHAR(255) NOT NULL COMMENT 'Original filename from upload',
    stored_filename VARCHAR(255) NOT NULL UNIQUE COMMENT 'Stored filename (sanitized + timestamp)',
    file_path VARCHAR(500) NOT NULL COMMENT 'Full path to file relative to tenant root',
    file_size BIGINT NOT NULL COMMENT 'File size in bytes',
    mime_type VARCHAR(100) NULL COMMENT 'MIME type (e.g., application/pdf)',
    file_hash VARCHAR(64) NULL COMMENT 'SHA-256 hash for duplicate detection',
    
    -- User Information
    uploaded_by INT NULL COMMENT 'User ID who uploaded the file',
    
    -- Relationship Information (polymorphic)
    related_id INT NULL COMMENT 'ID of related entity (student_id, payment_id, etc.)',
    related_type VARCHAR(50) NULL COMMENT 'Type of related entity (student, payment, report)',
    
    -- Metadata
    description TEXT NULL COMMENT 'Optional file description',
    tags VARCHAR(255) NULL COMMENT 'Comma-separated tags for searching',
    
    -- Access Control
    is_public TINYINT(1) DEFAULT 0 COMMENT 'Whether file is publicly accessible',
    access_count INT DEFAULT 0 COMMENT 'Number of times file was accessed',
    last_accessed_at DATETIME NULL COMMENT 'Last access timestamp',
    
    -- Timestamps
    upload_date DATETIME DEFAULT CURRENT_TIMESTAMP COMMENT 'When file was uploaded',
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'Last update',
    
    -- Soft Delete
    is_deleted TINYINT(1) DEFAULT 0 COMMENT 'Soft delete flag',
    deleted_at DATETIME NULL COMMENT 'When file was deleted',
    deleted_by INT NULL COMMENT 'User who deleted the file',
    
    -- Constraints
    CHECK (file_size >= 0),
    CHECK (access_count >= 0)
    
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Tracks all files uploaded by tenants';

-- =====================================================
-- STEP 4: CREATE INDEXES
-- =====================================================

-- Primary indexes for lookups
CREATE INDEX idx_tenant_id ON tenant_files(tenant_id);
CREATE INDEX idx_tenant_category ON tenant_files(tenant_id, file_category);
CREATE INDEX idx_stored_filename ON tenant_files(stored_filename);

-- Relationship indexes
CREATE INDEX idx_related ON tenant_files(related_type, related_id);
CREATE INDEX idx_uploaded_by ON tenant_files(uploaded_by);

-- Query optimization indexes
CREATE INDEX idx_upload_date ON tenant_files(upload_date);
CREATE INDEX idx_is_deleted ON tenant_files(is_deleted);
CREATE INDEX idx_tenant_not_deleted ON tenant_files(tenant_id, is_deleted);

-- Full-text search index (optional - for description and tags)
-- Uncomment if you need full-text search capabilities
-- ALTER TABLE tenant_files ADD FULLTEXT INDEX ft_search (description, tags);

-- =====================================================
-- STEP 5: ADD FOREIGN KEYS (OPTIONAL - COMMENT OUT IF NOT NEEDED)
-- =====================================================

-- Note: Foreign keys are optional and may cause issues if referenced tables don't exist
-- Uncomment these if you have the required tables and want referential integrity

-- Foreign key to tenants table
-- ALTER TABLE tenant_files
-- ADD CONSTRAINT fk_tenant_files_tenant 
-- FOREIGN KEY (tenant_id) 
-- REFERENCES tenants(academy_reference) 
-- ON DELETE CASCADE 
-- ON UPDATE CASCADE;

-- Foreign key to users table (if you have a unified users table)
-- ALTER TABLE tenant_files
-- ADD CONSTRAINT fk_tenant_files_uploaded_by 
-- FOREIGN KEY (uploaded_by) 
-- REFERENCES users(id) 
-- ON DELETE SET NULL 
-- ON UPDATE CASCADE;

-- Foreign key to users table for deleted_by
-- ALTER TABLE tenant_files
-- ADD CONSTRAINT fk_tenant_files_deleted_by 
-- FOREIGN KEY (deleted_by) 
-- REFERENCES users(id) 
-- ON DELETE SET NULL 
-- ON UPDATE CASCADE;

-- =====================================================
-- STEP 6: CREATE RELATED TABLES
-- =====================================================

-- File Access Log (tracks who accessed which files)
CREATE TABLE IF NOT EXISTS tenant_file_access_log (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    file_id INT NOT NULL,
    tenant_id CHAR(36) NOT NULL,
    user_id INT NULL,
    access_type ENUM('view', 'download', 'preview') DEFAULT 'view',
    ip_address VARCHAR(45) NULL,
    user_agent VARCHAR(255) NULL,
    accessed_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_file_id (file_id),
    INDEX idx_tenant_id (tenant_id),
    INDEX idx_user_id (user_id),
    INDEX idx_accessed_at (accessed_at),
    
    FOREIGN KEY (file_id) REFERENCES tenant_files(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Logs file access activity';

-- File Versions (for version control - optional)
CREATE TABLE IF NOT EXISTS tenant_file_versions (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    file_id INT NOT NULL,
    tenant_id CHAR(36) NOT NULL,
    version_number INT NOT NULL,
    stored_filename VARCHAR(255) NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    file_size BIGINT NOT NULL,
    uploaded_by INT NULL,
    version_notes TEXT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_file_id (file_id),
    INDEX idx_tenant_id (tenant_id),
    
    FOREIGN KEY (file_id) REFERENCES tenant_files(id) ON DELETE CASCADE,
    UNIQUE KEY unique_file_version (file_id, version_number)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Tracks file version history';

-- =====================================================
-- STEP 7: INSERT SAMPLE DATA (OPTIONAL - FOR TESTING)
-- =====================================================

-- Sample tenant files
INSERT INTO tenant_files (
    tenant_id, file_category, original_filename, stored_filename, 
    file_path, file_size, mime_type, uploaded_by, 
    related_id, related_type, description
) VALUES
-- Document samples
('soshigh_demo', 'document', 'student_report.pdf', 'student_report_1706600000.pdf', 
 'school_soshigh_demo/uploads/documents/student_report_1706600000.pdf', 
 245678, 'application/pdf', 1, 101, 'student', 'End of term report for student'),

('soshigh_demo', 'document', 'course_outline.docx', 'course_outline_1706600100.docx',
 'school_soshigh_demo/uploads/documents/course_outline_1706600100.docx',
 123456, 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 
 2, NULL, NULL, 'Mathematics course outline'),

-- Photo samples
('soshigh_demo', 'photo', 'john_doe.jpg', 'student_101_profile_1706600200.jpg',
 'school_soshigh_demo/uploads/profile_photos/student_101_profile_1706600200.jpg',
 89456, 'image/jpeg', 1, 101, 'student', 'Student profile photo'),

-- Receipt samples
('soshigh_demo', 'receipt', 'payment_receipt.pdf', 'payment_12345_receipt_1706600300.pdf',
 'school_soshigh_demo/uploads/payment_receipts/payment_12345_receipt_1706600300.pdf',
 56789, 'application/pdf', 1, 12345, 'payment', 'Tuition payment receipt'),

-- Report samples
('soshigh_demo', 'report', 'monthly_attendance.xlsx', 'attendance_report_202501_1706600400.xlsx',
 'school_soshigh_demo/uploads/reports/attendance_report_202501_1706600400.xlsx',
 234567, 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
 2, NULL, 'report', 'January 2025 attendance report');

-- =====================================================
-- STEP 8: VERIFICATION QUERIES
-- =====================================================

-- Check table was created
SELECT 'Table created successfully' AS status;

-- Show table structure
DESCRIBE tenant_files;

-- Show indexes
SHOW INDEXES FROM tenant_files;

-- Count records
SELECT COUNT(*) AS total_files FROM tenant_files;

-- Show sample data
SELECT 
    id, 
    tenant_id, 
    file_category, 
    original_filename, 
    file_size,
    upload_date
FROM tenant_files
LIMIT 5;

-- Storage by tenant
SELECT 
    tenant_id,
    COUNT(*) AS file_count,
    SUM(file_size) AS total_bytes,
    ROUND(SUM(file_size) / 1024 / 1024, 2) AS total_mb
FROM tenant_files
WHERE is_deleted = 0
GROUP BY tenant_id;

-- Files by category
SELECT 
    file_category,
    COUNT(*) AS count,
    ROUND(SUM(file_size) / 1024 / 1024, 2) AS total_mb
FROM tenant_files
WHERE is_deleted = 0
GROUP BY file_category
ORDER BY count DESC;

-- =====================================================
-- ADDITIONAL USEFUL QUERIES
-- =====================================================

-- Find duplicate files (by hash)
-- SELECT file_hash, COUNT(*) as duplicate_count
-- FROM tenant_files
-- WHERE file_hash IS NOT NULL
-- GROUP BY file_hash
-- HAVING COUNT(*) > 1;

-- Find largest files
-- SELECT 
--     tenant_id,
--     original_filename,
--     ROUND(file_size / 1024 / 1024, 2) AS size_mb,
--     upload_date
-- FROM tenant_files
-- WHERE is_deleted = 0
-- ORDER BY file_size DESC
-- LIMIT 10;

-- Files uploaded in last 7 days
-- SELECT 
--     tenant_id,
--     file_category,
--     COUNT(*) AS recent_files
-- FROM tenant_files
-- WHERE upload_date >= DATE_SUB(NOW(), INTERVAL 7 DAY)
-- AND is_deleted = 0
-- GROUP BY tenant_id, file_category;

-- =====================================================
-- STEP 9: CREATE TRIGGERS (OPTIONAL)
-- =====================================================

-- Trigger to update access count
DELIMITER $$

CREATE TRIGGER before_tenant_file_access_insert
AFTER INSERT ON tenant_file_access_log
FOR EACH ROW
BEGIN
    UPDATE tenant_files 
    SET 
        access_count = access_count + 1,
        last_accessed_at = NEW.accessed_at
    WHERE id = NEW.file_id;
END$$

DELIMITER ;

-- =====================================================
-- STEP 10: CREATE VIEWS (OPTIONAL)
-- =====================================================

-- View for active (non-deleted) files
CREATE OR REPLACE VIEW v_active_tenant_files AS
SELECT 
    id,
    tenant_id,
    file_category,
    original_filename,
    stored_filename,
    file_path,
    file_size,
    ROUND(file_size / 1024 / 1024, 2) AS size_mb,
    mime_type,
    uploaded_by,
    related_id,
    related_type,
    description,
    tags,
    is_public,
    access_count,
    upload_date,
    last_accessed_at
FROM tenant_files
WHERE is_deleted = 0;

-- View for file statistics by tenant
CREATE OR REPLACE VIEW v_tenant_file_stats AS
SELECT 
    tenant_id,
    COUNT(*) AS total_files,
    SUM(file_size) AS total_bytes,
    ROUND(SUM(file_size) / 1024 / 1024, 2) AS total_mb,
    ROUND(SUM(file_size) / 1024 / 1024 / 1024, 2) AS total_gb,
    MAX(upload_date) AS last_upload,
    SUM(CASE WHEN file_category = 'document' THEN 1 ELSE 0 END) AS documents,
    SUM(CASE WHEN file_category = 'photo' THEN 1 ELSE 0 END) AS photos,
    SUM(CASE WHEN file_category = 'receipt' THEN 1 ELSE 0 END) AS receipts,
    SUM(CASE WHEN file_category = 'report' THEN 1 ELSE 0 END) AS reports,
    SUM(CASE WHEN file_category = 'backup' THEN 1 ELSE 0 END) AS backups,
    SUM(CASE WHEN file_category = 'other' THEN 1 ELSE 0 END) AS other
FROM tenant_files
WHERE is_deleted = 0
GROUP BY tenant_id;

-- =====================================================
-- MIGRATION COMPLETE
-- =====================================================

SELECT '=== MIGRATION COMPLETE ===' AS status;
SELECT 'Tenant files tracking table created successfully' AS message;
SELECT 'Run rollback script if you need to undo changes' AS note;

-- =====================================================
-- END OF MIGRATION
-- =====================================================

