-- =====================================================
-- PAYMENT REMINDER & SUSPENSION SYSTEM
-- Multi-Tenant Payment Management Database Schema
-- Fixed version for production servers with restricted permissions
-- =====================================================

-- =====================================================
-- 1. TENANT PAYMENT SETTINGS
-- =====================================================
CREATE TABLE IF NOT EXISTS tenant_payment_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    academy_reference VARCHAR(50) NOT NULL,
    
    -- Payment cycle configuration
    payment_cycle_days INT NOT NULL DEFAULT 30 COMMENT 'Days between payments (default: monthly)',
    reminder_days_before INT NOT NULL DEFAULT 27 COMMENT 'Days after last payment to send reminder',
    warning_days_after_reminder INT NOT NULL DEFAULT 3 COMMENT 'Days after reminder to send warning',
    days_before_suspension INT NOT NULL DEFAULT 35 COMMENT 'Total days before suspension',
    
    -- Fee configuration
    monthly_fee DECIMAL(10,2) NOT NULL DEFAULT 0.00 COMMENT 'Monthly subscription fee per student',
    unsuspension_fee DECIMAL(10,2) NOT NULL DEFAULT 0.00 COMMENT 'Fee to unsuspend account',
    late_payment_fee DECIMAL(10,2) NOT NULL DEFAULT 0.00 COMMENT 'Additional late payment fee',
    
    -- Automation settings
    auto_suspend_enabled TINYINT(1) NOT NULL DEFAULT 1 COMMENT 'Enable automatic suspension',
    auto_reminder_enabled TINYINT(1) NOT NULL DEFAULT 1 COMMENT 'Enable automatic reminders',
    auto_warning_enabled TINYINT(1) NOT NULL DEFAULT 1 COMMENT 'Enable automatic warnings',
    
    -- Notification preferences
    send_email_notifications TINYINT(1) NOT NULL DEFAULT 1,
    send_sms_notifications TINYINT(1) NOT NULL DEFAULT 1,
    send_parent_notifications TINYINT(1) NOT NULL DEFAULT 1 COMMENT 'Also notify parents',
    
    -- Grace period settings
    grace_period_days INT NOT NULL DEFAULT 0 COMMENT 'Extra grace period before suspension',
    allow_partial_payments TINYINT(1) NOT NULL DEFAULT 0,
    
    -- Timestamps
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    created_by INT NULL COMMENT 'Admin ID who created settings',
    
    -- Indexes
    UNIQUE KEY idx_academy_reference (academy_reference),
    INDEX idx_auto_suspend (auto_suspend_enabled),
    INDEX idx_created_at (created_at),
    
    FOREIGN KEY (academy_reference) REFERENCES academy_references(reference_code) 
        ON UPDATE CASCADE ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Per-tenant payment and suspension configuration';

-- =====================================================
-- 2. STUDENT PAYMENT SCHEDULES
-- =====================================================
CREATE TABLE IF NOT EXISTS student_payment_schedules (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    academy_reference VARCHAR(50) NOT NULL,
    
    -- Payment dates
    last_payment_date DATE NULL COMMENT 'Date of last successful payment',
    next_due_date DATE NULL COMMENT 'Next payment due date',
    suspension_date DATE NULL COMMENT 'Date when account will be suspended',
    
    -- Payment status
    payment_status ENUM('active', 'pending', 'overdue', 'suspended', 'grace_period') NOT NULL DEFAULT 'active',
    total_amount_due DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    total_amount_paid DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    balance DECIMAL(10,2) NOT NULL DEFAULT 0.00,
    
    -- Reminder tracking
    reminder_sent_at TIMESTAMP NULL,
    reminder_count INT NOT NULL DEFAULT 0,
    warning_sent_at TIMESTAMP NULL,
    warning_count INT NOT NULL DEFAULT 0,
    last_notification_at TIMESTAMP NULL,
    
    -- Suspension tracking
    suspended_at TIMESTAMP NULL,
    suspension_reason VARCHAR(255) NULL,
    unsuspended_at TIMESTAMP NULL,
    is_manually_suspended TINYINT(1) NOT NULL DEFAULT 0,
    
    -- Timestamps
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    UNIQUE KEY idx_student_academy (student_id, academy_reference),
    INDEX idx_student_id (student_id),
    INDEX idx_academy_reference (academy_reference),
    INDEX idx_payment_status (payment_status),
    INDEX idx_next_due_date (next_due_date),
    INDEX idx_suspension_date (suspension_date),
    INDEX idx_last_payment_date (last_payment_date),
    INDEX idx_overdue (payment_status, next_due_date),
    
    FOREIGN KEY (student_id) REFERENCES students(id) 
        ON UPDATE CASCADE ON DELETE CASCADE,
    FOREIGN KEY (academy_reference) REFERENCES academy_references(reference_code) 
        ON UPDATE CASCADE ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Individual student payment tracking and schedules';

-- =====================================================
-- 3. PAYMENT REMINDERS LOG
-- =====================================================
CREATE TABLE IF NOT EXISTS payment_reminders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    academy_reference VARCHAR(50) NOT NULL,
    schedule_id INT NOT NULL COMMENT 'Reference to student_payment_schedules',
    
    -- Reminder details
    reminder_type ENUM('initial_reminder', 'warning', 'final_warning', 'suspension_notice') NOT NULL,
    sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    due_date DATE NOT NULL,
    amount_due DECIMAL(10,2) NOT NULL,
    
    -- Delivery status
    email_sent TINYINT(1) NOT NULL DEFAULT 0,
    email_sent_at TIMESTAMP NULL,
    email_error TEXT NULL,
    
    sms_sent TINYINT(1) NOT NULL DEFAULT 0,
    sms_sent_at TIMESTAMP NULL,
    sms_error TEXT NULL,
    
    parent_notified TINYINT(1) NOT NULL DEFAULT 0,
    parent_notified_at TIMESTAMP NULL,
    
    -- Payment link
    payment_link VARCHAR(500) NULL,
    payment_token VARCHAR(255) NULL,
    token_expires_at TIMESTAMP NULL,
    
    -- Response tracking
    link_clicked TINYINT(1) NOT NULL DEFAULT 0,
    link_clicked_at TIMESTAMP NULL,
    payment_completed TINYINT(1) NOT NULL DEFAULT 0,
    payment_completed_at TIMESTAMP NULL,
    
    -- Indexes
    INDEX idx_student_id (student_id),
    INDEX idx_academy_reference (academy_reference),
    INDEX idx_schedule_id (schedule_id),
    INDEX idx_reminder_type (reminder_type),
    INDEX idx_sent_at (sent_at),
    INDEX idx_due_date (due_date),
    INDEX idx_payment_token (payment_token),
    INDEX idx_pending_payment (payment_completed, sent_at),
    
    FOREIGN KEY (student_id) REFERENCES students(id) 
        ON UPDATE CASCADE ON DELETE CASCADE,
    FOREIGN KEY (academy_reference) REFERENCES academy_references(reference_code) 
        ON UPDATE CASCADE ON DELETE CASCADE,
    FOREIGN KEY (schedule_id) REFERENCES student_payment_schedules(id) 
        ON UPDATE CASCADE ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Log of all payment reminders sent to students';

-- =====================================================
-- 4. PAYMENT TRANSACTIONS (Enhanced)
-- =====================================================
CREATE TABLE IF NOT EXISTS payment_transactions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    academy_reference VARCHAR(50) NOT NULL,
    schedule_id INT NULL COMMENT 'Link to payment schedule',
    reminder_id INT NULL COMMENT 'Link to reminder that triggered payment',
    
    -- Transaction details
    transaction_reference VARCHAR(100) NOT NULL UNIQUE,
    external_reference VARCHAR(100) NULL COMMENT 'External payment gateway reference',
    amount DECIMAL(10,2) NOT NULL,
    currency VARCHAR(3) NOT NULL DEFAULT 'SZL',
    
    -- Payment method
    payment_method ENUM('momo', 'card', 'cash', 'bank_transfer', 'manual') NOT NULL,
    payment_provider VARCHAR(50) NULL COMMENT 'MTN, PayGate, etc.',
    
    -- Payment status
    status ENUM('pending', 'processing', 'completed', 'failed', 'refunded', 'cancelled') NOT NULL DEFAULT 'pending',
    
    -- Dates
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    paid_at TIMESTAMP NULL,
    failed_at TIMESTAMP NULL,
    refunded_at TIMESTAMP NULL,
    
    -- Payment details
    payer_phone VARCHAR(20) NULL,
    payer_email VARCHAR(255) NULL,
    payer_name VARCHAR(255) NULL,
    
    -- Response data
    raw_request TEXT NULL,
    raw_response TEXT NULL,
    error_message TEXT NULL,
    
    -- Processing
    processed TINYINT(1) NOT NULL DEFAULT 0,
    processed_at TIMESTAMP NULL,
    processed_by INT NULL COMMENT 'Admin ID if manually processed',
    
    -- Metadata
    description TEXT NULL,
    notes TEXT NULL,
    ip_address VARCHAR(45) NULL,
    user_agent VARCHAR(500) NULL,
    
    -- Indexes
    UNIQUE KEY idx_transaction_reference (transaction_reference),
    INDEX idx_student_id (student_id),
    INDEX idx_academy_reference (academy_reference),
    INDEX idx_schedule_id (schedule_id),
    INDEX idx_reminder_id (reminder_id),
    INDEX idx_external_reference (external_reference),
    INDEX idx_status (status),
    INDEX idx_payment_method (payment_method),
    INDEX idx_created_at (created_at),
    INDEX idx_paid_at (paid_at),
    INDEX idx_pending (status, created_at),
    
    FOREIGN KEY (student_id) REFERENCES students(id) 
        ON UPDATE CASCADE ON DELETE CASCADE,
    FOREIGN KEY (academy_reference) REFERENCES academy_references(reference_code) 
        ON UPDATE CASCADE ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='All payment transactions including recurring payments';

-- =====================================================
-- 5. SUSPENSION LOGS
-- =====================================================
CREATE TABLE IF NOT EXISTS suspension_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    academy_reference VARCHAR(50) NOT NULL,
    schedule_id INT NULL,
    
    -- Suspension details
    suspended_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    unsuspended_at TIMESTAMP NULL,
    duration_days INT NULL COMMENT 'Calculated suspension duration',
    
    -- Reason and type
    suspension_reason ENUM('payment_overdue', 'manual_admin', 'violation', 'other') NOT NULL DEFAULT 'payment_overdue',
    reason_description TEXT NULL,
    is_manual TINYINT(1) NOT NULL DEFAULT 0,
    
    -- Admin actions
    suspended_by INT NULL COMMENT 'Admin ID who suspended (if manual)',
    unsuspended_by INT NULL COMMENT 'Admin ID who unsuspended',
    admin_notes TEXT NULL,
    
    -- Moodle sync
    moodle_suspended TINYINT(1) NOT NULL DEFAULT 0,
    moodle_suspended_at TIMESTAMP NULL,
    moodle_unsuspended TINYINT(1) NOT NULL DEFAULT 0,
    moodle_unsuspended_at TIMESTAMP NULL,
    moodle_sync_error TEXT NULL,
    
    -- Notifications
    student_notified TINYINT(1) NOT NULL DEFAULT 0,
    parent_notified TINYINT(1) NOT NULL DEFAULT 0,
    notification_sent_at TIMESTAMP NULL,
    
    -- Payment to unsuspend
    unsuspension_payment_required DECIMAL(10,2) NULL,
    unsuspension_payment_id INT NULL COMMENT 'Payment transaction ID',
    
    -- Status
    status ENUM('active', 'lifted', 'expired') NOT NULL DEFAULT 'active',
    
    -- Indexes
    INDEX idx_student_id (student_id),
    INDEX idx_academy_reference (academy_reference),
    INDEX idx_schedule_id (schedule_id),
    INDEX idx_suspended_at (suspended_at),
    INDEX idx_status (status),
    INDEX idx_suspension_reason (suspension_reason),
    INDEX idx_is_manual (is_manual),
    INDEX idx_active_suspensions (status, suspended_at),
    INDEX idx_moodle_pending (moodle_suspended, status),
    
    FOREIGN KEY (student_id) REFERENCES students(id) 
        ON UPDATE CASCADE ON DELETE CASCADE,
    FOREIGN KEY (academy_reference) REFERENCES academy_references(reference_code) 
        ON UPDATE CASCADE ON DELETE CASCADE,
    FOREIGN KEY (schedule_id) REFERENCES student_payment_schedules(id) 
        ON UPDATE CASCADE ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Complete history of all account suspensions and unsuspensions';

-- =====================================================
-- 6. PAYMENT NOTIFICATION TEMPLATES
-- =====================================================
CREATE TABLE IF NOT EXISTS payment_notification_templates (
    id INT AUTO_INCREMENT PRIMARY KEY,
    academy_reference VARCHAR(50) NOT NULL,
    
    -- Template identification
    template_type ENUM('reminder', 'warning', 'suspension', 'payment_success', 'unsuspension') NOT NULL,
    notification_channel ENUM('email', 'sms', 'both') NOT NULL,
    
    -- Template content
    subject VARCHAR(255) NULL COMMENT 'Email subject line',
    email_body TEXT NULL COMMENT 'HTML email template',
    sms_body TEXT NULL COMMENT 'SMS message template (160 chars)',
    
    -- Settings
    is_active TINYINT(1) NOT NULL DEFAULT 1,
    is_default TINYINT(1) NOT NULL DEFAULT 0,
    
    -- Metadata
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    created_by INT NULL,
    
    -- Indexes
    INDEX idx_academy_reference (academy_reference),
    INDEX idx_template_type (template_type),
    INDEX idx_is_active (is_active),
    
    FOREIGN KEY (academy_reference) REFERENCES academy_references(reference_code) 
        ON UPDATE CASCADE ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Customizable notification templates per tenant';

-- =====================================================
-- 7. PAYMENT SYSTEM AUDIT LOG
-- =====================================================
CREATE TABLE IF NOT EXISTS payment_system_audit_log (
    id INT AUTO_INCREMENT PRIMARY KEY,
    
    -- Who and when
    user_id INT NULL COMMENT 'Admin/user who performed action',
    user_type ENUM('admin', 'student', 'parent', 'system') NOT NULL DEFAULT 'system',
    academy_reference VARCHAR(50) NOT NULL,
    
    -- What happened
    action VARCHAR(100) NOT NULL,
    entity_type VARCHAR(50) NOT NULL COMMENT 'student, payment, suspension, etc.',
    entity_id INT NULL,
    
    -- Details
    old_values JSON NULL,
    new_values JSON NULL,
    description TEXT NULL,
    
    -- Context
    ip_address VARCHAR(45) NULL,
    user_agent VARCHAR(500) NULL,
    request_url VARCHAR(500) NULL,
    
    -- Timestamp
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_user_id (user_id),
    INDEX idx_academy_reference (academy_reference),
    INDEX idx_action (action),
    INDEX idx_entity (entity_type, entity_id),
    INDEX idx_created_at (created_at),
    
    FOREIGN KEY (academy_reference) REFERENCES academy_references(reference_code) 
        ON UPDATE CASCADE ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Audit trail for all payment system actions';

-- =====================================================
-- 8. ADD SUSPENSION COLUMNS TO STUDENTS TABLE
-- =====================================================
ALTER TABLE students 
ADD COLUMN IF NOT EXISTS suspended TINYINT(1) NOT NULL DEFAULT 0 AFTER email_verified,
ADD COLUMN IF NOT EXISTS suspended_at TIMESTAMP NULL AFTER suspended,
ADD COLUMN IF NOT EXISTS suspension_reason VARCHAR(255) NULL AFTER suspended_at;

-- Add index if it doesn't exist
-- Note: MySQL doesn't support IF NOT EXISTS for indexes, so we ignore errors
-- CREATE INDEX idx_suspended ON students(suspended);

-- =====================================================
-- INSERT DEFAULT SETTINGS FOR EXISTING TENANTS
-- =====================================================

INSERT INTO tenant_payment_settings (
    academy_reference, 
    payment_cycle_days, 
    reminder_days_before,
    warning_days_after_reminder,
    days_before_suspension,
    monthly_fee,
    unsuspension_fee,
    auto_suspend_enabled,
    auto_reminder_enabled
)
SELECT 
    reference_code,
    30, -- Monthly cycle
    27, -- Remind 27 days after last payment
    3,  -- Warning 3 days after reminder
    35, -- Suspend at 35 days
    350.00, -- Default monthly fee (adjust per tenant)
    50.00,  -- Unsuspension fee
    1,  -- Auto-suspend enabled
    1   -- Auto-reminders enabled
FROM academy_references
WHERE is_active = 1
  AND reference_code NOT IN (SELECT academy_reference FROM tenant_payment_settings)
ON DUPLICATE KEY UPDATE academy_reference = academy_references.reference_code;

-- =====================================================
-- CREATE STORED PROCEDURES
-- =====================================================

DELIMITER //

-- Procedure: Initialize payment schedule for new student
DROP PROCEDURE IF EXISTS sp_init_student_payment_schedule//
CREATE PROCEDURE sp_init_student_payment_schedule(
    IN p_student_id INT,
    IN p_academy_reference VARCHAR(50)
)
BEGIN
    DECLARE v_monthly_fee DECIMAL(10,2);
    DECLARE v_payment_cycle_days INT;
    
    -- Get tenant settings
    SELECT COALESCE(monthly_fee, 350.00), COALESCE(payment_cycle_days, 30)
    INTO v_monthly_fee, v_payment_cycle_days
    FROM tenant_payment_settings
    WHERE academy_reference = p_academy_reference
    LIMIT 1;
    
    -- Create schedule
    INSERT INTO student_payment_schedules (
        student_id,
        academy_reference,
        last_payment_date,
        next_due_date,
        payment_status,
        total_amount_due,
        balance
    ) VALUES (
        p_student_id,
        p_academy_reference,
        CURDATE(),
        DATE_ADD(CURDATE(), INTERVAL v_payment_cycle_days DAY),
        'active',
        v_monthly_fee,
        v_monthly_fee
    )
    ON DUPLICATE KEY UPDATE
        last_payment_date = CURDATE(),
        next_due_date = DATE_ADD(CURDATE(), INTERVAL v_payment_cycle_days DAY),
        payment_status = 'active';
END//

-- Procedure: Process successful payment
DROP PROCEDURE IF EXISTS sp_process_payment_success//
CREATE PROCEDURE sp_process_payment_success(
    IN p_transaction_id INT
)
BEGIN
    DECLARE v_student_id INT;
    DECLARE v_amount DECIMAL(10,2);
    DECLARE v_schedule_id INT;
    DECLARE v_academy_reference VARCHAR(50);
    DECLARE v_payment_cycle_days INT;
    
    -- Get transaction details
    SELECT student_id, amount, schedule_id, academy_reference
    INTO v_student_id, v_amount, v_schedule_id, v_academy_reference
    FROM payment_transactions
    WHERE id = p_transaction_id;
    
    -- Get payment cycle
    SELECT COALESCE(payment_cycle_days, 30)
    INTO v_payment_cycle_days
    FROM tenant_payment_settings
    WHERE academy_reference = v_academy_reference
    LIMIT 1;
    
    -- Update payment schedule
    UPDATE student_payment_schedules
    SET last_payment_date = CURDATE(),
        next_due_date = DATE_ADD(CURDATE(), INTERVAL v_payment_cycle_days DAY),
        payment_status = 'active',
        total_amount_paid = total_amount_paid + v_amount,
        balance = GREATEST(0, balance - v_amount),
        suspended_at = NULL,
        suspension_reason = NULL,
        is_manually_suspended = 0,
        reminder_sent_at = NULL,
        warning_sent_at = NULL,
        updated_at = CURRENT_TIMESTAMP
    WHERE id = v_schedule_id;
    
    -- Update suspension log if exists
    UPDATE suspension_logs
    SET unsuspended_at = CURRENT_TIMESTAMP,
        status = 'lifted',
        unsuspension_payment_id = p_transaction_id
    WHERE student_id = v_student_id
      AND status = 'active';
      
    -- Mark transaction as processed
    UPDATE payment_transactions
    SET processed = 1,
        processed_at = CURRENT_TIMESTAMP
    WHERE id = p_transaction_id;
END//

DELIMITER ;

-- =====================================================
-- COMPLETION MESSAGE
-- =====================================================

SELECT 'Payment Reminder System Schema Created Successfully!' AS Status;

SELECT 'Tables Created' AS Info, COUNT(*) AS Count
FROM information_schema.tables 
WHERE table_schema = DATABASE()
  AND table_name IN (
    'tenant_payment_settings',
    'student_payment_schedules',
    'payment_reminders',
    'payment_transactions',
    'suspension_logs',
    'payment_notification_templates',
    'payment_system_audit_log'
  );

