-- =====================================================
-- CLEANUP OLD SUBJECTS & PREPARE FOR RE-IMPORT
-- =====================================================
-- This script removes old/duplicate subjects and prepares
-- the database for a clean import of Eswatini curriculum
-- =====================================================

-- =====================================================
-- OPTION 1: CLEAN SLATE (Remove all subjects and mappings)
-- =====================================================
-- Use this if you want to start completely fresh

-- WARNING: This will delete ALL subjects and grade mappings!
-- Uncomment the lines below to execute:

-- -- Disable foreign key checks temporarily
-- SET FOREIGN_KEY_CHECKS = 0;
-- 
-- -- Delete all grade-subject mappings
-- TRUNCATE TABLE grade_subject;
-- 
-- -- Delete all subjects
-- TRUNCATE TABLE subjects;
-- 
-- -- Re-enable foreign key checks
-- SET FOREIGN_KEY_CHECKS = 1;
-- 
-- -- Verify deletion
-- SELECT COUNT(*) as subjects_remaining FROM subjects;
-- SELECT COUNT(*) as mappings_remaining FROM grade_subject;

-- =====================================================
-- OPTION 2: SELECTIVE CLEANUP (Remove only Eswatini subjects)
-- =====================================================
-- Use this if you have other subjects you want to keep

-- Backup first
CREATE TABLE IF NOT EXISTS subjects_before_cleanup AS SELECT * FROM subjects;
CREATE TABLE IF NOT EXISTS grade_subject_before_cleanup AS SELECT * FROM grade_subject;

-- Remove grade mappings for Eswatini subjects
DELETE gs FROM grade_subject gs
JOIN subjects s ON gs.subject_id = s.id
WHERE s.code LIKE 'G4-%'
   OR s.code LIKE 'G5-%'
   OR s.code LIKE 'G6-%'
   OR s.code LIKE 'G7-%'
   OR s.code LIKE 'F1-%'
   OR s.code LIKE 'F2-%'
   OR s.code LIKE 'F3-%'
   OR s.code LIKE 'F4-%'
   OR s.code LIKE 'F5-%';

-- Remove Eswatini subjects
DELETE FROM subjects
WHERE code LIKE 'G4-%'
   OR code LIKE 'G5-%'
   OR code LIKE 'G6-%'
   OR code LIKE 'G7-%'
   OR code LIKE 'F1-%'
   OR code LIKE 'F2-%'
   OR code LIKE 'F3-%'
   OR code LIKE 'F4-%'
   OR code LIKE 'F5-%';

-- Verify cleanup
SELECT 
    'Eswatini subjects remaining' as check_item,
    COUNT(*) as count
FROM subjects
WHERE code LIKE 'G4-%'
   OR code LIKE 'G5-%'
   OR code LIKE 'G6-%'
   OR code LIKE 'G7-%'
   OR code LIKE 'F1-%'
   OR code LIKE 'F2-%'
   OR code LIKE 'F3-%'
   OR code LIKE 'F4-%'
   OR code LIKE 'F5-%';

-- =====================================================
-- OPTION 3: REMOVE ONLY SUBJECTS WITH INVALID/DUPLICATE CODES
-- =====================================================

-- Find subjects with duplicate codes
SELECT 
    code,
    COUNT(*) as count,
    GROUP_CONCAT(id) as duplicate_ids
FROM subjects
GROUP BY code
HAVING COUNT(*) > 1;

-- Delete duplicates (keeps the first occurrence)
-- Uncomment to execute:
-- DELETE s1 FROM subjects s1
-- INNER JOIN subjects s2 
-- WHERE s1.id > s2.id 
-- AND s1.code = s2.code;

-- =====================================================
-- VERIFICATION AFTER CLEANUP
-- =====================================================

SELECT '=== CLEANUP VERIFICATION ===' as section;

-- Total subjects
SELECT COUNT(*) as total_subjects FROM subjects;

-- Subjects by type
SELECT 
    CASE 
        WHEN code LIKE 'G4-%' OR code LIKE 'G5-%' OR 
             code LIKE 'G6-%' OR code LIKE 'G7-%' THEN 'Primary (G4-G7)'
        WHEN code LIKE 'F1-%' OR code LIKE 'F2-%' OR 
             code LIKE 'F3-%' THEN 'Junior Secondary (F1-F3)'
        WHEN code LIKE 'F4-%' OR code LIKE 'F5-%' THEN 'Senior Secondary (F4-F5)'
        ELSE 'Other/Legacy'
    END as category,
    COUNT(*) as subject_count
FROM subjects
GROUP BY category;

-- Grade mappings
SELECT 
    g.name as grade,
    COUNT(gs.subject_id) as linked_subjects
FROM grades g
LEFT JOIN grade_subject gs ON g.id = gs.grade_id
GROUP BY g.name
ORDER BY g.name;

-- =====================================================
-- READY FOR IMPORT
-- =====================================================

SELECT '=== STATUS ===' as section;

SELECT 
    CASE 
        WHEN (SELECT COUNT(*) FROM subjects WHERE code LIKE 'G4-%') > 0 
        THEN '⚠️ Eswatini subjects still exist. Run cleanup option again.'
        ELSE '✅ Database is clean. Ready for fresh import of ESWATINI_CURRICULUM_SUBJECTS.sql'
    END as import_readiness;

-- =====================================================
-- NEXT STEPS
-- =====================================================

SELECT '=== NEXT STEPS ===' as next_action;

SELECT '1. Run: database/migrations/fix_subjects_duplicates.sql' as step_1;
SELECT '2. Run: database/verify_subjects_fix.sql' as step_2;
SELECT '3. Run: database/ESWATINI_CURRICULUM_SUBJECTS.sql' as step_3;
SELECT '4. Verify with: SELECT COUNT(*) FROM subjects' as step_4;
SELECT '5. Map subjects to Moodle courses in admin panel' as step_5;

-- =====================================================
-- ROLLBACK (IF NEEDED)
-- =====================================================
-- To restore data before cleanup:
-- 
-- TRUNCATE TABLE subjects;
-- INSERT INTO subjects SELECT * FROM subjects_before_cleanup;
-- 
-- TRUNCATE TABLE grade_subject;
-- INSERT INTO grade_subject SELECT * FROM grade_subject_before_cleanup;
-- 
-- =====================================================

