<?php
/**
 * Daily Payment Status Check Cron Job
 * Runs daily to check payment statuses and send appropriate emails
 */

require_once __DIR__ . '/cron_utils.php';
require_once __DIR__ . '/../includes/email_templates.php';

$logFile = __DIR__ . '/../logs/daily_payment_check.log';

logMessage("=== DAILY PAYMENT CHECK STARTED ===");

try {
    $pdo = getCronDB();
    if (!$pdo) {
        throw new Exception("Database connection failed");
    }
    
    // Get current date
    $today = date('Y-m-d');
    $todayDateTime = date('Y-m-d H:i:s');
    
    logMessage("Checking payment statuses for: $today");
    
    // 1. Check for students who need payment reminders (26 days before due date)
    $reminderDate = date('Y-m-d', strtotime('+26 days', strtotime($today)));
    $reminderStudents = getStudentsForReminder($reminderDate);
    
    foreach ($reminderStudents as $student) {
        echo "Sending payment reminder to: {$student['first_name']} {$student['last_name']}\n";
        sendPaymentReminder($student);
    }
    
    // 2. Check for students who need warning emails (30 days overdue)
    $warningDate = date('Y-m-d', strtotime('+30 days', strtotime($today)));
    $warningStudents = getStudentsForWarning($warningDate);
    
    foreach ($warningStudents as $student) {
        echo "Sending payment warning to: {$student['first_name']} {$student['last_name']}\n";
        sendPaymentWarning($student);
    }
    
    // 3. Check for students who need suspension (33 days overdue)
    $suspensionDate = date('Y-m-d', strtotime('+33 days', strtotime($today)));
    $suspensionStudents = getStudentsForSuspension($suspensionDate);
    
    foreach ($suspensionStudents as $student) {
        echo "Suspending student: {$student['first_name']} {$student['last_name']}\n";
        suspendStudent($student);
    }
    
    // 4. Update payment cycles for students who paid
    updatePaymentCycles();
    
    echo "[" . date('Y-m-d H:i:s') . "] Daily payment check completed successfully.\n";
    
} catch (Exception $e) {
    echo "[" . date('Y-m-d H:i:s') . "] ERROR: " . $e->getMessage() . "\n";
    error_log("Daily payment check error: " . $e->getMessage());
}

function getStudentsForReminder($reminderDate) {
    global $pdo;
    
    $sql = "
        SELECT s.*, 
               (SELECT COALESCE(SUM(sb.price), 0) FROM student_subject ss 
                JOIN subjects sb ON ss.subject_id = sb.id 
                WHERE ss.student_id = s.id) as amount_due
        FROM students s 
        WHERE s.next_payment_due = ? 
        AND s.payment_reminder_sent = 0
        AND s.email_verified = 1
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$reminderDate]);
    return $stmt->fetchAll();
}

function getStudentsForWarning($warningDate) {
    global $pdo;
    
    $sql = "
        SELECT s.*, 
               (SELECT COALESCE(SUM(sb.price), 0) FROM student_subject ss 
                JOIN subjects sb ON ss.subject_id = sb.id 
                WHERE ss.student_id = s.id) as amount_due
        FROM students s 
        WHERE s.next_payment_due = ? 
        AND s.warning_email_sent = 0
        AND s.email_verified = 1
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$warningDate]);
    return $stmt->fetchAll();
}

function getStudentsForSuspension($suspensionDate) {
    global $pdo;
    
    $sql = "
        SELECT s.*, 
               (SELECT COALESCE(SUM(sb.price), 0) FROM student_subject ss 
                JOIN subjects sb ON ss.subject_id = sb.id 
                WHERE ss.student_id = s.id) as amount_due
        FROM students s 
        WHERE s.next_payment_due = ? 
        AND s.suspension_warning_sent = 0
        AND s.email_verified = 1
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$suspensionDate]);
    return $stmt->fetchAll();
}

function sendPaymentReminder($student) {
    global $pdo;
    
    // Get student's subjects
    $subjects = getSubjectsForStudent($student['id']);
    $amountDue = $student['amount_due'];
    
    // Generate payment link
    $paymentLink = getBaseUrl() . "/payments/pay.php?student_id=" . $student['id'] . "&reminder=1";
    
    // Get email template
    $emailData = getPaymentReminderEmail($student, $amountDue, $paymentLink, $subjects);
    
    // Send email
    $mail = new PHPMailer(true);
    try {
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_SECURE;
        $mail->Port = SMTP_PORT;
        
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($student['email'], $student['first_name'] . ' ' . $student['last_name']);
        
        $mail->isHTML(true);
        $mail->Subject = $emailData['subject'];
        $mail->Body = $emailData['html'];
        
        $mail->send();
        
        // Mark reminder as sent
        $stmt = $pdo->prepare('UPDATE students SET payment_reminder_sent = 1 WHERE id = ?');
        $stmt->execute([$student['id']]);
        
        echo "✓ Payment reminder sent to {$student['email']}\n";
        
    } catch (Exception $e) {
        echo "✗ Failed to send reminder to {$student['email']}: " . $e->getMessage() . "\n";
    }
}

function sendPaymentWarning($student) {
    global $pdo;
    
    // Get student's subjects
    $subjects = getSubjectsForStudent($student['id']);
    $amountDue = $student['amount_due'];
    
    // Generate payment link
    $paymentLink = getBaseUrl() . "/payments/pay.php?student_id=" . $student['id'] . "&warning=1";
    
    // Get email template
    $emailData = getPaymentWarningEmail($student, $amountDue, $paymentLink, $subjects);
    
    // Send email
    $mail = new PHPMailer(true);
    try {
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_SECURE;
        $mail->Port = SMTP_PORT;
        
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($student['email'], $student['first_name'] . ' ' . $student['last_name']);
        
        $mail->isHTML(true);
        $mail->Subject = $emailData['subject'];
        $mail->Body = $emailData['html'];
        
        $mail->send();
        
        // Mark warning as sent
        $stmt = $pdo->prepare('UPDATE students SET warning_email_sent = 1 WHERE id = ?');
        $stmt->execute([$student['id']]);
        
        echo "✓ Payment warning sent to {$student['email']}\n";
        
    } catch (Exception $e) {
        echo "✗ Failed to send warning to {$student['email']}: " . $e->getMessage() . "\n";
    }
}

function suspendStudent($student) {
    global $pdo;
    
    try {
        // Update student status in database
        $stmt = $pdo->prepare('UPDATE students SET moodle_status = "suspended", suspended_date = NOW() WHERE id = ?');
        $stmt->execute([$student['id']]);
        
        // Get student's subjects for suspension email
        $subjects = getSubjectsForStudent($student['id']);
        $amountDue = $student['amount_due'];
        
        // Generate payment link
        $paymentLink = getBaseUrl() . "/payments/pay.php?student_id=" . $student['id'] . "&suspended=1";
        
        // Send suspension email
        $emailData = getSuspensionEmail($student, $amountDue, $paymentLink, $subjects);
        
        $mail = new PHPMailer(true);
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_SECURE;
        $mail->Port = SMTP_PORT;
        
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        $mail->addAddress($student['email'], $student['first_name'] . ' ' . $student['last_name']);
        
        $mail->isHTML(true);
        $mail->Subject = $emailData['subject'];
        $mail->Body = $emailData['html'];
        
        $mail->send();
        
        // Mark suspension warning as sent
        $stmt = $pdo->prepare('UPDATE students SET suspension_warning_sent = 1 WHERE id = ?');
        $stmt->execute([$student['id']]);
        
        // Trigger bot to suspend in Moodle
        triggerMoodleSyncBot();
        
        echo "✓ Student suspended and notification sent to {$student['email']}\n";
        
    } catch (Exception $e) {
        echo "✗ Failed to suspend student {$student['id']}: " . $e->getMessage() . "\n";
    }
}

function updatePaymentCycles() {
    global $pdo;
    
    // Update payment cycles for students who have paid
    $sql = "
        UPDATE students s 
        SET 
            last_payment_date = NOW(),
            next_payment_due = DATE_ADD(NOW(), INTERVAL 30 DAY),
            payment_reminder_sent = 0,
            warning_email_sent = 0,
            suspension_warning_sent = 0,
            moodle_status = 'active',
            suspended_date = NULL
        WHERE s.id IN (
            SELECT DISTINCT p.student_id 
            FROM payments p 
            WHERE p.status = 'SUCCESSFUL' 
            AND p.created_at > s.last_payment_date
        )
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute();
    $affected = $stmt->rowCount();
    
    if ($affected > 0) {
        echo "✓ Updated payment cycles for $affected students\n";
    }
}

function getBaseUrl() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $path = dirname($_SERVER['SCRIPT_NAME'] ?? '');
    return $protocol . '://' . $host . $path;
}
?>
