#!/usr/bin/env python3
"""
Bot API for Dashboard Integration
Provides real-time bot status and metrics for the admin dashboard
"""

import json
import os
import sys
from pathlib import Path
from datetime import datetime, timedelta
import re
from typing import Dict, List, Any

# Add the bot directory to the path
sys.path.append(str(Path(__file__).parent))

from config import LOG_FILES, LOCK_FILE

class BotAPI:
    """API class for providing bot data to the dashboard"""
    
    def __init__(self):
        self.log_file = LOG_FILES['sync']
        self.lock_file = LOCK_FILE
    
    def get_status(self) -> Dict[str, Any]:
        """Get current bot status"""
        status = {
            'status': 'idle',
            'last_run': None,
            'next_run': None,
            'lock_file': False,
            'log_size': 0,
            'error_count': 0,
            'bot_type': 'python',
            'started_at': None
        }
        
        # Check if bot is running (lock file exists)
        if self.lock_file.exists():
            status['lock_file'] = True
            status['status'] = 'running'
            status['started_at'] = datetime.fromtimestamp(
                self.lock_file.stat().st_mtime
            ).strftime('%Y-%m-%d %H:%M:%S')
        
        # Get log file info
        if self.log_file.exists():
            status['log_size'] = self.log_file.stat().st_size
            
            # Read last few lines to find last completion
            with open(self.log_file, 'r', encoding='utf-8') as f:
                lines = f.readlines()
            
            # Reverse to get most recent first
            lines.reverse()
            
            for line in lines:
                if '=== Moodle Sync Bot Completed Successfully ===' in line:
                    # Extract timestamp - handle format with milliseconds
                    timestamp_match = re.search(r'\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}),\d+\]', line)
                    if timestamp_match:
                        status['last_run'] = timestamp_match.group(1)
                    break
                elif '=== Moodle Sync Bot Started ===' in line:
                    # Also check for started entries
                    timestamp_match = re.search(r'\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}),\d+\]', line)
                    if timestamp_match and not status['last_run']:
                        status['last_run'] = timestamp_match.group(1)
            
            # Count recent errors (last 50 lines)
            error_count = 0
            recent_lines = lines[:50]
            for line in recent_lines:
                if '[ERROR]' in line:
                    error_count += 1
            status['error_count'] = error_count
        
        # Estimate next run (assuming cron runs every minute)
        if status['last_run']:
            try:
                last_run = datetime.strptime(status['last_run'], '%Y-%m-%d %H:%M:%S')
                next_run = last_run + timedelta(minutes=1)
                status['next_run'] = next_run.strftime('%Y-%m-%d %H:%M:%S')
            except ValueError:
                pass
        
        return status
    
    def get_logs(self, lines: int = 50) -> Dict[str, Any]:
        """Get recent log entries"""
        if not self.log_file.exists():
            return {'logs': [], 'message': 'No log file found'}
        
        with open(self.log_file, 'r', encoding='utf-8') as f:
            log_lines = f.readlines()
        
        # Reverse to get most recent first
        log_lines.reverse()
        log_lines = log_lines[:lines]
        
        logs = []
        for line in log_lines:
            logs.append({
                'line': line.strip(),
                'timestamp': self._extract_timestamp(line),
                'level': self._extract_log_level(line),
                'message': self._extract_message(line)
            })
        
        return {'logs': logs}
    
    def get_metrics(self) -> Dict[str, Any]:
        """Get bot performance metrics"""
        metrics = {
            'total_syncs': 0,
            'successful_syncs': 0,
            'failed_syncs': 0,
            'total_users_uploaded': 0,
            'total_enrollments': 0,
            'average_sync_time': 0,
            'last_7_days': []
        }
        
        if not self.log_file.exists():
            return metrics
        
        with open(self.log_file, 'r', encoding='utf-8') as f:
            lines = f.readlines()
        
        sync_times = []
        latest_users_uploaded = 0
        latest_enrollments = 0
        
        for line in lines:
            # Count sync attempts
            if '=== Moodle Sync Bot Started ===' in line:
                metrics['total_syncs'] += 1
            
            # Count successful syncs
            if '=== Moodle Sync Bot Completed Successfully ===' in line:
                metrics['successful_syncs'] += 1
            
            # Count failed syncs
            if '=== Moodle Sync Bot Failed ===' in line:
                metrics['failed_syncs'] += 1
            
            # Extract sync time
            if 'Sync completed in' in line:
                time_match = re.search(r'Sync completed in ([\d.]+) seconds', line)
                if time_match:
                    sync_times.append(float(time_match.group(1)))
            
            # Count users uploaded - get the LATEST count, not accumulate
            if 'Successfully uploaded' in line and 'new users' in line:
                users_match = re.search(r'Successfully uploaded (\d+) new users', line)
                if users_match:
                    latest_users_uploaded = int(users_match.group(1))
            
            # Count enrollments - get the LATEST count from enrollment summary
            if 'Enrollment completed:' in line:
                enroll_match = re.search(r'Enrollment completed: (\d+) enrollments', line)
                if enroll_match:
                    latest_enrollments = int(enroll_match.group(1))
            
            # Also check for individual successful enrollments in recent logs
            if '[SUCCESS] Enrolled user' in line:
                # Count individual successes but don't override the summary count
                pass
        
        # Use the latest counts, not accumulated
        metrics['total_users_uploaded'] = latest_users_uploaded
        metrics['total_enrollments'] = latest_enrollments
        
        if sync_times:
            metrics['average_sync_time'] = round(sum(sync_times) / len(sync_times), 2)
        
        return metrics
    
    def _extract_timestamp(self, line: str) -> str:
        """Extract timestamp from log line"""
        timestamp_match = re.search(r'\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}),\d+\]', line)
        return timestamp_match.group(1) if timestamp_match else None
    
    def _extract_log_level(self, line: str) -> str:
        """Extract log level from log line"""
        if '[ERROR]' in line:
            return 'error'
        elif '[WARNING]' in line:
            return 'warning'
        elif '[INFO]' in line:
            return 'info'
        else:
            return 'info'
    
    def _extract_message(self, line: str) -> str:
        """Extract message from log line"""
        # Remove timestamp and log level (handle milliseconds format)
        message = re.sub(r'\[\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2},\d+\] \[(ERROR|WARNING|INFO)\]', '', line)
        return message.strip()

def main():
    """Main function for CGI execution"""
    import cgi
    
    # Set up CGI
    form = cgi.FieldStorage()
    action = form.getvalue('action', 'status')
    
    # Create API instance
    api = BotAPI()
    
    # Set content type
    print("Content-Type: application/json\n")
    
    try:
        if action == 'status':
            result = api.get_status()
        elif action == 'logs':
            lines = int(form.getvalue('lines', 50))
            result = api.get_logs(lines)
        elif action == 'metrics':
            result = api.get_metrics()
        else:
            result = {'error': 'Invalid action'}
        
        print(json.dumps(result, indent=2))
        
    except Exception as e:
        print(json.dumps({'error': str(e)}, indent=2))

if __name__ == "__main__":
    main()
