# Python Moodle Sync Bot

A Python-based bot that synchronizes student data from Skolo-Kine to Moodle. This is a Python conversion of the original PHP bot with enhanced functionality and better error handling.

## Features

- **CSV Download**: Downloads student data from Skolo-Kine API
- **Data Parsing**: Parses and validates student information
- **Moodle Integration**: Creates users and enrolls them in courses
- **Error Handling**: Comprehensive error handling and logging
- **Web Interface**: Web-based monitoring and control interface
- **Testing**: Built-in testing functionality
- **Retry Logic**: Automatic retry mechanism for failed operations

## Requirements

- Python 3.7 or higher
- MySQL database (same as PHP version)
- Moodle instance with API access
- Required Python packages (see requirements.txt)

## Installation

### 1. Clone or Download

Place the bot files in your `bot/` directory alongside the existing PHP bot.

### 2. Install Dependencies

```bash
cd bot
python3 setup.py
```

Or manually install requirements:

```bash
pip install -r requirements.txt
```

### 3. Configure

Edit `config.py` to match your environment:

```python
# Database configuration
DB_CONFIG = {
    'host': '127.0.0.1',
    'port': 3306,
    'database': 'skolo_kine',
    'user': 'root',
    'password': 'your_password',  # Update this
    'charset': 'utf8mb4'
}

# Moodle configuration
MOODLE_CONFIG = {
    'api_url': 'https://your-moodle.com/webservice/rest/server.php',
    'api_token': 'your_api_token'  # Update this
}
```

### 4. Test Installation

```bash
python3 test_user_creation.py
```

## Usage

### Manual Execution

```bash
python3 moodle_sync_bot.py
```

### Web Interface

Access the web interface at:
```
http://localhost/Skolo-Kine/bot/web_interface.py
```

### Automated Execution (Cron)

Add to your crontab for every minute execution:

```bash
*/1 * * * * /usr/bin/python3 /path/to/bot/moodle_sync_bot.py
```

## File Structure

```
bot/
├── config.py              # Configuration settings
├── database.py            # Database utility functions
├── moodle_sync_bot.py     # Main bot script
├── test_user_creation.py  # Testing script
├── web_interface.py       # Web interface
├── setup.py              # Setup script
├── requirements.txt      # Python dependencies
├── README.md            # This file
├── temp/                # Temporary files
└── logs/                # Log files
    ├── sync.log
    ├── test_user_creation.log
    └── test_sync.log
```

## Configuration

### Database Settings

The bot uses the same MySQL database as the PHP version. Ensure your database credentials are correct in `config.py`.

### Moodle API

1. Enable web services in Moodle
2. Create a service with the following functions:
   - `core_user_create_users`
   - `core_user_get_users_by_field`
   - `core_course_get_courses`
   - `enrol_manual_enrol_users`
   - `core_webservice_get_site_info`
3. Create a token with appropriate permissions
4. Update the token in `config.py`

### Subject Mapping

The bot automatically maps subjects to Moodle courses based on course names. You can customize the mapping in `config.py`:

```python
COURSE_NAME_PATTERNS = {
    'mathematics': 2,
    'math': 2,
    'english': 5,
    'science': 18,
    # Add more patterns as needed
}
```

## Logging

The bot creates detailed logs in the `logs/` directory:

- `sync.log`: Main bot execution logs
- `test_user_creation.log`: Test execution logs
- `test_sync.log`: Sync test logs

Log levels:
- **INFO**: General information
- **WARNING**: Non-critical issues
- **ERROR**: Critical errors

## Error Handling

The bot includes comprehensive error handling:

1. **Connection Errors**: Retries with exponential backoff
2. **API Errors**: Detailed error logging and reporting
3. **Data Validation**: Validates all user data before upload
4. **Lock File**: Prevents concurrent executions

## Testing

### Test User Creation

```bash
python3 test_user_creation.py
```

This tests:
- Moodle API connection
- User creation permissions
- Sample user creation
- CSV download functionality

### Test Database Connection

```python
from database import test_database_connection
test_database_connection()
```

## Troubleshooting

### Common Issues

1. **Database Connection Failed**
   - Check database credentials in `config.py`
   - Ensure MySQL server is running
   - Verify database exists

2. **Moodle API Errors**
   - Check API token permissions
   - Verify Moodle web services are enabled
   - Check Moodle URL and token

3. **CSV Download Failed**
   - Verify Skolo-Kine API URL
   - Check API key
   - Ensure network connectivity

4. **Permission Errors**
   - Ensure bot has write permissions to `temp/` and `logs/` directories
   - Check file ownership

### Debug Mode

Enable debug mode in `config.py`:

```python
APP_CONFIG = {
    'debug': True
}
```

## Migration from PHP

The Python bot is designed to be a drop-in replacement for the PHP bot:

1. **Same Database**: Uses the same MySQL database
2. **Same API**: Uses the same Skolo-Kine and Moodle APIs
3. **Same Functionality**: All features from PHP version included
4. **Enhanced Features**: Better error handling and logging

### Migration Steps

1. Install Python bot alongside PHP bot
2. Test Python bot thoroughly
3. Update cron jobs to use Python version
4. Monitor logs for any issues
5. Remove PHP bot when confident

## Performance

The Python bot includes several performance optimizations:

- **Connection Pooling**: Reuses HTTP connections
- **Batch Processing**: Processes users in configurable batches
- **Efficient Parsing**: Optimized CSV parsing
- **Memory Management**: Proper resource cleanup

## Security

Security features include:

- **Input Validation**: All user data is validated
- **SQL Injection Prevention**: Uses parameterized queries
- **API Key Protection**: Secure API key handling
- **Error Information**: Limited error details in production

## Contributing

To contribute to the Python bot:

1. Fork the repository
2. Create a feature branch
3. Make your changes
4. Add tests
5. Submit a pull request

## License

This project maintains the same license as the original PHP version.

## Support

For support and questions:

1. Check the logs for error details
2. Review the troubleshooting section
3. Test individual components
4. Contact the development team

## Changelog

### Version 1.0.0
- Initial Python conversion
- All PHP functionality implemented
- Enhanced error handling
- Web interface
- Comprehensive testing
- Documentation
