<?php
/**
 * API Endpoint: Delete File
 * 
 * DELETE /api/storage/file/{id}
 * 
 * Soft deletes a file from tenant storage
 */

header('Content-Type: application/json');

session_start();
require_once '../../config.php';
require_once '../../includes/tenant_file_upload.php';
require_once '../../includes/tenant_directory_manager.php';

// Check authentication
if (!isset($_SESSION['user_id']) || !isset($_SESSION['academy_reference'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

// Only handle DELETE requests
if ($_SERVER['REQUEST_METHOD'] !== 'DELETE') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// Get file ID from URL
$file_id = $_GET['id'] ?? null;

if (!$file_id) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing file_id parameter']);
    exit;
}

try {
    // Database connection
    $conn = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
    
    // Get file info
    $stmt = $conn->prepare("
        SELECT tenant_id, file_path 
        FROM tenant_files 
        WHERE id = ? AND is_deleted = 0
    ");
    $stmt->execute([$file_id]);
    $file = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$file) {
        http_response_code(404);
        echo json_encode(['error' => 'File not found or already deleted']);
        exit;
    }
    
    // Check permission (can only delete own tenant's files)
    if ($file['tenant_id'] !== $_SESSION['academy_reference'] && $_SESSION['role'] !== 'super_admin') {
        http_response_code(403);
        echo json_encode(['error' => 'Access denied']);
        exit;
    }
    
    // Initialize upload handler
    $dirManager = new TenantDirectoryManager($conn);
    $uploader = new TenantFileUpload($conn, $dirManager);
    
    // Delete file
    $result = $uploader->deleteFile($file['tenant_id'], $file['file_path']);
    
    if ($result['success']) {
        echo json_encode([
            'success' => true,
            'message' => 'File deleted successfully',
            'file_id' => $file_id
        ]);
    } else {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => $result['error']
        ]);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

