<?php
/**
 * School Settings Management
 * Each school can customize their settings
 */

require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
requireAdminLogin();
requireSchool();

$currentSchool = getCurrentSchool();
$pdo = getDB();

$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $settings = [
        'school_name' => trim($_POST['school_name'] ?? ''),
        'school_email' => trim($_POST['school_email'] ?? ''),
        'school_phone' => trim($_POST['school_phone'] ?? ''),
        'school_address' => trim($_POST['school_address'] ?? ''),
        'primary_color' => trim($_POST['primary_color'] ?? '#4F46E5'),
        'secondary_color' => trim($_POST['secondary_color'] ?? '#10B981'),
        'academic_year' => trim($_POST['academic_year'] ?? ''),
        'email_from_name' => trim($_POST['email_from_name'] ?? ''),
        'email_signature' => trim($_POST['email_signature'] ?? ''),
        'moodle_url' => trim($_POST['moodle_url'] ?? '')
    ];
    
    // Validation
    if (empty($settings['school_name'])) $errors[] = 'School name is required';
    if (!empty($settings['school_email']) && !filter_var($settings['school_email'], FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Invalid email address';
    }
    
    if (empty($errors)) {
        try {
            $pdo->beginTransaction();
            
            foreach ($settings as $key => $value) {
                setSchoolSetting($key, $value);
            }
            
            // Update academy name in main table
            $stmt = $pdo->prepare("
                UPDATE academy_references 
                SET academy_name = ?, updated_at = NOW()
                WHERE reference_code = ?
            ");
            $stmt->execute([$settings['school_name'], $currentSchool]);
            
            $pdo->commit();
            $success = true;
            
            // Clear cache
            unset($_SESSION['school_data_cache']);
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $errors[] = 'Error saving settings: ' . $e->getMessage();
        }
    }
}

// Get current settings
$currentSettings = [];
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$currentSchool]);
while ($row = $stmt->fetch()) {
    $currentSettings[$row['setting_key']] = $row['setting_value'];
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>School Settings - <?= getSchoolName() ?></title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="../../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../../images/img-melanesys-logo.svg">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800&display=swap" rel="stylesheet">
    
    <style>
        * {
            font-family: 'Inter', sans-serif;
        }
        
        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
        }
        
        .settings-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 3rem 0 4rem;
            margin-bottom: -2rem;
            position: relative;
        }
        
        .settings-header::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            height: 100px;
            background: inherit;
            clip-path: ellipse(60% 60px at 50% 100%);
        }
        
        .settings-container {
            max-width: 900px;
            margin: 0 auto;
        }
        
        .settings-card {
            border: none;
            border-radius: 20px;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.08);
            margin-bottom: 2rem;
            overflow: hidden;
            transition: transform 0.3s, box-shadow 0.3s;
        }
        
        .settings-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 50px rgba(0, 0, 0, 0.12);
        }
        
        .settings-card-header {
            background: linear-gradient(135deg, #667eea15, #764ba215);
            border-bottom: 3px solid #667eea;
            padding: 1.5rem;
            font-weight: 700;
            font-size: 1.2rem;
            color: #1a1a1a;
        }
        
        .settings-card-header i {
            color: #667eea;
            margin-right: 0.75rem;
            font-size: 1.5rem;
        }
        
        .form-label {
            font-weight: 600;
            color: #374151;
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .form-control, .form-select {
            border-radius: 12px;
            border: 2px solid #e5e7eb;
            padding: 0.75rem 1rem;
            transition: all 0.3s;
        }
        
        .form-control:focus, .form-select:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .color-preview {
            width: 60px;
            height: 60px;
            border-radius: 12px;
            border: 3px solid #e5e7eb;
            cursor: pointer;
            transition: all 0.3s;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        
        .color-preview:hover {
            transform: scale(1.1);
            box-shadow: 0 6px 20px rgba(0, 0, 0, 0.15);
        }
        
        .form-control-color {
            height: 60px;
            border-radius: 12px;
            border: 3px solid #e5e7eb;
            cursor: pointer;
        }
        
        .btn-save {
            background: linear-gradient(135deg, #667eea, #764ba2);
            border: none;
            color: white;
            padding: 1rem 3rem;
            border-radius: 50px;
            font-weight: 700;
            font-size: 1.1rem;
            transition: all 0.3s;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        }
        
        .btn-save:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
            color: white;
        }
        
        .btn-cancel {
            background: white;
            border: 2px solid #e5e7eb;
            color: #6b7280;
            padding: 1rem 3rem;
            border-radius: 50px;
            font-weight: 600;
            font-size: 1.1rem;
            transition: all 0.3s;
        }
        
        .btn-cancel:hover {
            border-color: #9ca3af;
            background: #f9fafb;
            color: #374151;
        }
        
        .alert-modern {
            border: none;
            border-radius: 15px;
            padding: 1.25rem 1.5rem;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
        }
        
        .alert-success {
            background: linear-gradient(135deg, #d1fae515, #a7f3d015);
            border-left: 4px solid #10b981;
            color: #065f46;
        }
        
        .alert-danger {
            background: linear-gradient(135deg, #fee2e215, #fecaca15);
            border-left: 4px solid #ef4444;
            color: #991b1b;
        }
        
        .info-box {
            background: linear-gradient(135deg, #dbeafe15, #bfdbfe15);
            border-left: 4px solid #3b82f6;
            border-radius: 12px;
            padding: 1.25rem;
            margin: 1rem 0;
        }
        
        .color-picker-group {
            display: flex;
            align-items: center;
            gap: 1rem;
        }
        
        .color-label {
            flex: 1;
        }
        
        .breadcrumb {
            background: transparent;
            padding: 0;
            margin-bottom: 1rem;
        }
        
        .breadcrumb-item + .breadcrumb-item::before {
            color: rgba(255, 255, 255, 0.6);
        }
        
        .breadcrumb-item a {
            color: rgba(255, 255, 255, 0.9);
            text-decoration: none;
        }
        
        .breadcrumb-item.active {
            color: white;
        }
        
        .form-text {
            color: #6b7280;
            font-size: 0.85rem;
            margin-top: 0.5rem;
        }
        
        .section-divider {
            border: none;
            height: 2px;
            background: linear-gradient(90deg, transparent, #e5e7eb, transparent);
            margin: 2rem 0;
        }
    </style>
</head>
<body>
    <?php include '../partials/header.php'; ?>

    <!-- Header Section -->
    <div class="settings-header">
        <div class="container settings-container">
            <nav aria-label="breadcrumb">
                <ol class="breadcrumb">
                    <li class="breadcrumb-item"><a href="../dashboard.php"><i class="bi bi-house me-1"></i>Dashboard</a></li>
                    <li class="breadcrumb-item active">School Settings</li>
                </ol>
            </nav>
            <h1 class="display-5 fw-bold mb-2">
                <i class="bi bi-gear-fill me-3"></i>School Settings
            </h1>
            <p class="lead mb-0 opacity-75">Customize your school's appearance and information</p>
        </div>
    </div>

    <div class="container settings-container py-4">
        <?php if ($success): ?>
            <div class="alert alert-modern alert-success fade show mb-4">
                <div class="d-flex align-items-center">
                    <i class="bi bi-check-circle-fill me-3 fs-3"></i>
                    <div>
                        <strong>Success!</strong>
                        <p class="mb-0">Your school settings have been saved successfully.</p>
                    </div>
                </div>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (!empty($errors)): ?>
            <div class="alert alert-modern alert-danger mb-4">
                <div class="d-flex align-items-start">
                    <i class="bi bi-exclamation-triangle-fill me-3 fs-3"></i>
                    <div class="flex-grow-1">
                        <strong>Please fix the following errors:</strong>
                        <ul class="mb-0 mt-2">
                            <?php foreach ($errors as $error): ?>
                                <li><?= htmlspecialchars($error) ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <form method="POST">
            <!-- General Information -->
            <div class="settings-card">
                <div class="settings-card-header">
                    <i class="bi bi-info-circle-fill"></i>General Information
                </div>
                <div class="card-body p-4">
                            <div class="mb-3">
                                <label for="school_name" class="form-label">School Name <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="school_name" name="school_name" 
                                       value="<?= htmlspecialchars($currentSettings['school_name'] ?? $school['academy_name']) ?>" required>
                            </div>

                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="school_email" class="form-label">School Email</label>
                                    <input type="email" class="form-control" id="school_email" name="school_email" 
                                           value="<?= htmlspecialchars($currentSettings['school_email'] ?? '') ?>">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="school_phone" class="form-label">School Phone</label>
                                    <input type="text" class="form-control" id="school_phone" name="school_phone" 
                                           value="<?= htmlspecialchars($currentSettings['school_phone'] ?? '') ?>">
                                </div>
                            </div>

                            <div class="mb-3">
                                <label for="school_address" class="form-label">School Address</label>
                                <textarea class="form-control" id="school_address" name="school_address" rows="2"><?= htmlspecialchars($currentSettings['school_address'] ?? '') ?></textarea>
                            </div>

                            <div class="mb-3">
                                <label for="academic_year" class="form-label">Academic Year</label>
                                <input type="text" class="form-control" id="academic_year" name="academic_year" 
                                       value="<?= htmlspecialchars($currentSettings['academic_year'] ?? date('Y') . '/' . (date('Y')+1)) ?>"
                                       placeholder="2024/2025">
                            </div>
                        </div>
                    </div>

            <!-- Branding -->
            <div class="settings-card">
                <div class="settings-card-header">
                    <i class="bi bi-palette-fill"></i>Branding & Colors
                </div>
                <div class="card-body p-4">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="primary_color" class="form-label">Primary Color</label>
                                    <input type="color" class="form-control form-control-color" id="primary_color" 
                                           name="primary_color" value="<?= htmlspecialchars($currentSettings['primary_color'] ?? '#4F46E5') ?>">
                                    <small class="text-muted">Used for buttons, headers, etc.</small>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="secondary_color" class="form-label">Secondary Color</label>
                                    <input type="color" class="form-control form-control-color" id="secondary_color" 
                                           name="secondary_color" value="<?= htmlspecialchars($currentSettings['secondary_color'] ?? '#10B981') ?>">
                                    <small class="text-muted">Used for accents, highlights, etc.</small>
                                </div>
                            </div>

                            <div class="info-box">
                                <div class="d-flex align-items-center">
                                    <i class="bi bi-lightbulb-fill me-3 fs-4 text-primary"></i>
                                    <div>
                                        <strong>Logo Upload</strong>
                                        <p class="mb-0 small mt-1">Contact super admin to upload a custom logo for your school</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

            <!-- Email Settings -->
            <div class="settings-card">
                <div class="settings-card-header">
                    <i class="bi bi-envelope-fill"></i>Email Configuration
                </div>
                <div class="card-body p-4">
                            <div class="mb-3">
                                <label for="email_from_name" class="form-label">Email From Name</label>
                                <input type="text" class="form-control" id="email_from_name" name="email_from_name" 
                                       value="<?= htmlspecialchars($currentSettings['email_from_name'] ?? $currentSettings['school_name'] ?? '') ?>"
                                       placeholder="e.g., Kine Academy">
                                <small class="text-muted">Name that appears as sender in emails</small>
                            </div>

                            <div class="mb-3">
                                <label for="email_signature" class="form-label">Email Signature</label>
                                <textarea class="form-control" id="email_signature" name="email_signature" rows="4"
                                          placeholder="Best regards,&#10;Kine Academy Team"><?= htmlspecialchars($currentSettings['email_signature'] ?? '') ?></textarea>
                                <small class="text-muted">Automatically added to all outgoing emails</small>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Moodle Integration Settings -->
                    <div class="card shadow-sm mb-4">
                        <div class="card-header bg-info text-white">
                            <h5 class="mb-0"><i class="bi bi-mortarboard me-2"></i>Moodle Integration</h5>
                        </div>
                        <div class="card-body">
                            <div class="mb-3">
                                <label for="moodle_url" class="form-label">
                                    <i class="bi bi-link-45deg me-1"></i>Moodle Site URL
                                </label>
                                <input type="url" class="form-control" id="moodle_url" name="moodle_url" 
                                       value="<?= htmlspecialchars($currentSettings['moodle_url'] ?? '') ?>"
                                       placeholder="https://moodle.yourschool.com">
                                <small class="text-muted">
                                    <i class="bi bi-info-circle me-1"></i>
                                    This URL will be included in student credential emails with a direct link to access Moodle.
                                    Students can use their same login credentials for both the portal and Moodle.
                                </small>
                            </div>
                            
                            <div class="alert alert-info mb-0">
                                <i class="bi bi-lightbulb me-2"></i>
                                <strong>Tip:</strong> Make sure to also configure Moodle subject mappings in the 
                                <a href="../moodle/configure_mapping.php" class="alert-link">Moodle Configuration</a> page 
                                for automatic student enrollment.
                            </div>
                        </div>
                    </div>

            <div class="section-divider"></div>
            
            <div class="d-flex gap-3 justify-content-center pb-4">
                <button type="submit" class="btn-save">
                    <i class="bi bi-check-circle-fill me-2"></i>Save Settings
                </button>
                <a href="../dashboard.php" class="btn-cancel">
                    <i class="bi bi-x-circle me-2"></i>Cancel
                </a>
            </div>
        </form>
    </div>

    <?php include '../partials/footer.php'; ?>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Color picker preview updates
        document.querySelectorAll('input[type="color"]').forEach(picker => {
            picker.addEventListener('change', function() {
                this.style.transform = 'scale(1.1)';
                setTimeout(() => {
                    this.style.transform = 'scale(1)';
                }, 300);
            });
        });
        
        // Form animation on load
        document.querySelectorAll('.settings-card').forEach((card, index) => {
            card.style.animation = `fadeInUp 0.6s ease ${index * 0.1}s backwards`;
        });
        
        const style = document.createElement('style');
        style.textContent = `
            @keyframes fadeInUp {
                from {
                    opacity: 0;
                    transform: translateY(20px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
        `;
        document.head.appendChild(style);
    </script>
</body>
</html>

