<?php
/**
 * Payment Settings Management
 * Configure payment reminders and suspension settings per tenant
 */

session_start();
require_once '../../includes/functions.php';
require_once '../../includes/school_context.php';
require_once '../../includes/favicon_helper.php';
requireAdminLogin();

$pdo = getDB();
$academy_reference = getCurrentSchool();
$success = '';
$error = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_settings'])) {
    try {
        $stmt = $pdo->prepare("
            INSERT INTO tenant_payment_settings (
                academy_reference, payment_cycle_days, reminder_days_before,
                warning_days_after_reminder, days_before_suspension,
                monthly_fee, unsuspension_fee, late_payment_fee,
                auto_suspend_enabled, auto_reminder_enabled, auto_warning_enabled,
                send_email_notifications, send_sms_notifications, send_parent_notifications,
                grace_period_days, allow_partial_payments, created_by
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE
                payment_cycle_days = VALUES(payment_cycle_days),
                reminder_days_before = VALUES(reminder_days_before),
                warning_days_after_reminder = VALUES(warning_days_after_reminder),
                days_before_suspension = VALUES(days_before_suspension),
                monthly_fee = VALUES(monthly_fee),
                unsuspension_fee = VALUES(unsuspension_fee),
                late_payment_fee = VALUES(late_payment_fee),
                auto_suspend_enabled = VALUES(auto_suspend_enabled),
                auto_reminder_enabled = VALUES(auto_reminder_enabled),
                auto_warning_enabled = VALUES(auto_warning_enabled),
                send_email_notifications = VALUES(send_email_notifications),
                send_sms_notifications = VALUES(send_sms_notifications),
                send_parent_notifications = VALUES(send_parent_notifications),
                grace_period_days = VALUES(grace_period_days),
                allow_partial_payments = VALUES(allow_partial_payments)
        ");
        
        $stmt->execute([
            $academy_reference,
            (int)$_POST['payment_cycle_days'],
            (int)$_POST['reminder_days_before'],
            (int)$_POST['warning_days_after_reminder'],
            (int)$_POST['days_before_suspension'],
            (float)$_POST['monthly_fee'],
            (float)$_POST['unsuspension_fee'],
            (float)$_POST['late_payment_fee'],
            isset($_POST['auto_suspend_enabled']) ? 1 : 0,
            isset($_POST['auto_reminder_enabled']) ? 1 : 0,
            isset($_POST['auto_warning_enabled']) ? 1 : 0,
            isset($_POST['send_email_notifications']) ? 1 : 0,
            isset($_POST['send_sms_notifications']) ? 1 : 0,
            isset($_POST['send_parent_notifications']) ? 1 : 0,
            (int)$_POST['grace_period_days'],
            isset($_POST['allow_partial_payments']) ? 1 : 0,
            $_SESSION['admin_id']
        ]);
        
        $success = 'Payment settings saved successfully!';
    } catch (Exception $e) {
        $error = 'Error saving settings: ' . $e->getMessage();
    }
}

// Load current settings
$stmt = $pdo->prepare("
    SELECT * FROM tenant_payment_settings WHERE academy_reference = ?
");
$stmt->execute([$academy_reference]);
$settings = $stmt->fetch() ?: [
    'payment_cycle_days' => 30,
    'reminder_days_before' => 27,
    'warning_days_after_reminder' => 3,
    'days_before_suspension' => 35,
    'monthly_fee' => 350.00,
    'unsuspension_fee' => 50.00,
    'late_payment_fee' => 0.00,
    'auto_suspend_enabled' => 1,
    'auto_reminder_enabled' => 1,
    'auto_warning_enabled' => 1,
    'send_email_notifications' => 1,
    'send_sms_notifications' => 1,
    'send_parent_notifications' => 1,
    'grace_period_days' => 0,
    'allow_partial_payments' => 0
];

// Get payment statistics
$stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total_students,
        SUM(CASE WHEN payment_status = 'active' THEN 1 ELSE 0 END) as active_students,
        SUM(CASE WHEN payment_status = 'overdue' THEN 1 ELSE 0 END) as overdue_students,
        SUM(CASE WHEN payment_status = 'suspended' THEN 1 ELSE 0 END) as suspended_students,
        SUM(CASE WHEN DATEDIFF(CURDATE(), last_payment_date) >= ? THEN 1 ELSE 0 END) as need_reminders
    FROM student_payment_schedules
    WHERE academy_reference = ?
");
$stmt->execute([$settings['reminder_days_before'], $academy_reference]);
$stats = $stmt->fetch();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Settings - <?= APP_NAME ?></title>
    <?php echoFavicon($academy_reference); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        .settings-card {
            border-radius: 15px;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        .section-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px 20px;
            border-radius: 15px 15px 0 0;
            font-weight: bold;
        }
        .stat-card {
            text-align: center;
            padding: 20px;
            border-radius: 10px;
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        .stat-number {
            font-size: 32px;
            font-weight: bold;
            margin: 10px 0;
        }
        .form-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            margin: 15px 0;
        }
    </style>
</head>
<body class="bg-light">
    <div class="container mt-4 mb-5">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h2><i class="bi bi-gear-fill text-primary"></i> Payment Settings</h2>
                <p class="text-muted mb-0">Configure payment reminders and suspension rules</p>
            </div>
            <a href="../dashboard.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Back to Dashboard
            </a>
        </div>

        <?php if ($success): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <i class="bi bi-check-circle-fill"></i> <?= htmlspecialchars($success) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <?php if ($error): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bi bi-exclamation-triangle-fill"></i> <?= htmlspecialchars($error) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <!-- Statistics -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="stat-card">
                    <i class="bi bi-people text-primary" style="font-size: 24px;"></i>
                    <div class="stat-number text-primary"><?= $stats['total_students'] ?></div>
                    <small class="text-muted">Total Students</small>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card">
                    <i class="bi bi-check-circle text-success" style="font-size: 24px;"></i>
                    <div class="stat-number text-success"><?= $stats['active_students'] ?></div>
                    <small class="text-muted">Active Payments</small>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card">
                    <i class="bi bi-exclamation-circle text-warning" style="font-size: 24px;"></i>
                    <div class="stat-number text-warning"><?= $stats['overdue_students'] ?></div>
                    <small class="text-muted">Overdue Payments</small>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card">
                    <i class="bi bi-ban text-danger" style="font-size: 24px;"></i>
                    <div class="stat-number text-danger"><?= $stats['suspended_students'] ?></div>
                    <small class="text-muted">Suspended</small>
                </div>
            </div>
        </div>

        <!-- Settings Form -->
        <div class="card settings-card">
            <div class="section-header">
                <i class="bi bi-sliders"></i> Payment Configuration
            </div>
            <div class="card-body p-4">
                <form method="POST">
                    
                    <!-- Payment Cycle Settings -->
                    <div class="form-section">
                        <h5 class="mb-3"><i class="bi bi-calendar-event text-primary"></i> Payment Cycle</h5>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Payment Cycle (Days)</label>
                                <input type="number" class="form-control" name="payment_cycle_days" 
                                       value="<?= $settings['payment_cycle_days'] ?>" min="1" required>
                                <small class="text-muted">How often students need to pay (e.g., 30 days = monthly)</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Monthly Fee (SZL)</label>
                                <input type="number" class="form-control" name="monthly_fee" 
                                       value="<?= $settings['monthly_fee'] ?>" step="0.01" min="0" required>
                                <small class="text-muted">Amount charged per payment cycle</small>
                            </div>
                        </div>
                    </div>

                    <!-- Reminder Timeline -->
                    <div class="form-section">
                        <h5 class="mb-3"><i class="bi bi-bell text-warning"></i> Reminder Timeline</h5>
                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Send Reminder After (Days)</label>
                                <input type="number" class="form-control" name="reminder_days_before" 
                                       value="<?= $settings['reminder_days_before'] ?>" min="1" required>
                                <small class="text-muted">Days after last payment to send reminder</small>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Send Warning After (Days)</label>
                                <input type="number" class="form-control" name="warning_days_after_reminder" 
                                       value="<?= $settings['warning_days_after_reminder'] ?>" min="1" required>
                                <small class="text-muted">Days after reminder to send warning</small>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Suspend Account After (Days)</label>
                                <input type="number" class="form-control" name="days_before_suspension" 
                                       value="<?= $settings['days_before_suspension'] ?>" min="1" required>
                                <small class="text-muted">Total days before suspension</small>
                            </div>
                        </div>
                        
                        <!-- Timeline Visualization -->
                        <div class="alert alert-info">
                            <strong><i class="bi bi-info-circle"></i> Timeline Preview:</strong><br>
                            Day 0: Last payment → 
                            Day <?= $settings['reminder_days_before'] ?>: Reminder sent → 
                            Day <?= $settings['reminder_days_before'] + $settings['warning_days_after_reminder'] ?>: Warning sent → 
                            Day <?= $settings['days_before_suspension'] ?>: Account suspended
                        </div>
                    </div>

                    <!-- Fees -->
                    <div class="form-section">
                        <h5 class="mb-3"><i class="bi bi-currency-dollar text-success"></i> Additional Fees</h5>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Unsuspension Fee (SZL)</label>
                                <input type="number" class="form-control" name="unsuspension_fee" 
                                       value="<?= $settings['unsuspension_fee'] ?>" step="0.01" min="0">
                                <small class="text-muted">Fee to reactivate suspended account (0 for none)</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Late Payment Fee (SZL)</label>
                                <input type="number" class="form-control" name="late_payment_fee" 
                                       value="<?= $settings['late_payment_fee'] ?>" step="0.01" min="0">
                                <small class="text-muted">Additional fee for late payments (0 for none)</small>
                            </div>
                        </div>
                    </div>

                    <!-- Automation Settings -->
                    <div class="form-section">
                        <h5 class="mb-3"><i class="bi bi-robot text-info"></i> Automation</h5>
                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" name="auto_reminder_enabled" 
                                           <?= $settings['auto_reminder_enabled'] ? 'checked' : '' ?>>
                                    <label class="form-check-label">
                                        <strong>Auto Send Reminders</strong><br>
                                        <small class="text-muted">Automatically send payment reminders</small>
                                    </label>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" name="auto_warning_enabled" 
                                           <?= $settings['auto_warning_enabled'] ? 'checked' : '' ?>>
                                    <label class="form-check-label">
                                        <strong>Auto Send Warnings</strong><br>
                                        <small class="text-muted">Automatically send warning notices</small>
                                    </label>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" name="auto_suspend_enabled" 
                                           <?= $settings['auto_suspend_enabled'] ? 'checked' : '' ?>>
                                    <label class="form-check-label">
                                        <strong>Auto Suspend Accounts</strong><br>
                                        <small class="text-muted">Automatically suspend overdue accounts</small>
                                    </label>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Notification Channels -->
                    <div class="form-section">
                        <h5 class="mb-3"><i class="bi bi-send text-primary"></i> Notification Channels</h5>
                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" name="send_email_notifications" 
                                           <?= $settings['send_email_notifications'] ? 'checked' : '' ?>>
                                    <label class="form-check-label">
                                        <strong>Email Notifications</strong><br>
                                        <small class="text-muted">Send via email</small>
                                    </label>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" name="send_sms_notifications" 
                                           <?= $settings['send_sms_notifications'] ? 'checked' : '' ?>>
                                    <label class="form-check-label">
                                        <strong>SMS Notifications</strong><br>
                                        <small class="text-muted">Send via SMS</small>
                                    </label>
                                </div>
                            </div>
                            <div class="col-md-4 mb-3">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" name="send_parent_notifications" 
                                           <?= $settings['send_parent_notifications'] ? 'checked' : '' ?>>
                                    <label class="form-check-label">
                                        <strong>Notify Parents</strong><br>
                                        <small class="text-muted">Also notify parents/guardians</small>
                                    </label>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Additional Options -->
                    <div class="form-section">
                        <h5 class="mb-3"><i class="bi bi-toggles text-secondary"></i> Additional Options</h5>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Grace Period (Days)</label>
                                <input type="number" class="form-control" name="grace_period_days" 
                                       value="<?= $settings['grace_period_days'] ?>" min="0">
                                <small class="text-muted">Extra days before suspension (added to timeline)</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <div class="form-check form-switch mt-4">
                                    <input class="form-check-input" type="checkbox" name="allow_partial_payments" 
                                           <?= $settings['allow_partial_payments'] ? 'checked' : '' ?>>
                                    <label class="form-check-label">
                                        <strong>Allow Partial Payments</strong><br>
                                        <small class="text-muted">Students can pay partial amounts</small>
                                    </label>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Submit Button -->
                    <div class="d-grid gap-2 mt-4">
                        <button type="submit" name="save_settings" class="btn btn-primary btn-lg">
                            <i class="bi bi-save"></i> Save Payment Settings
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="row mt-4">
            <div class="col-md-4">
                <div class="card">
                    <div class="card-body text-center">
                        <i class="bi bi-people" style="font-size: 48px; color: #667eea;"></i>
                        <h5 class="mt-3">Manage Students</h5>
                        <p class="text-muted">View payment status and manage suspensions</p>
                        <a href="manage_student_payments.php" class="btn btn-primary">
                            <i class="bi bi-arrow-right"></i> View Students
                        </a>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card">
                    <div class="card-body text-center">
                        <i class="bi bi-mortarboard" style="font-size: 48px; color: #28a745;"></i>
                        <h5 class="mt-3">Moodle Setup</h5>
                        <p class="text-muted">Configure Moodle integration</p>
                        <a href="../moodle/index.php" class="btn btn-success">
                            <i class="bi bi-arrow-right"></i> Moodle Hub
                        </a>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card">
                    <div class="card-body text-center">
                        <i class="bi bi-clipboard-check" style="font-size: 48px; color: #17a2b8;"></i>
                        <h5 class="mt-3">Test System</h5>
                        <p class="text-muted">Test Moodle integration</p>
                        <a href="../../test_moodle_complete_workflow.php?key=test123" class="btn btn-info">
                            <i class="bi bi-arrow-right"></i> Run Tests
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

