<?php
/**
 * Simple Student Export Handler  
 * Works without external libraries
 * Supports: CSV, HTML with Letterhead
 */

session_start();
require_once '../includes/functions.php';
require_once '../includes/school_context.php';

// Require admin login
requireAdminLogin();

// Get school from session or redirect
$currentSchool = getCurrentSchool();
if (!$currentSchool) {
    header('Location: ../select_school.php');
    exit;
}

$schoolName = getSchoolName();
$pdo = getDB();

// Get school settings
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$currentSchool]);
$schoolSettings = [];
while ($row = $stmt->fetch()) {
    $schoolSettings[$row['setting_key']] = $row['setting_value'];
}

// Get school details
$stmt = $pdo->prepare("SELECT * FROM academy_references WHERE reference_code = ?");
$stmt->execute([$currentSchool]);
$schoolData = $stmt->fetch();

$format = $_GET['format'] ?? 'csv';

// Build query (same as dashboard filters)
$filterGrade = (int)($_GET['grade'] ?? 0);
$filterSubject = (int)($_GET['subject'] ?? 0);
$filterEmailStatus = trim($_GET['email_status'] ?? '');
$filterDateFrom = trim($_GET['date_from'] ?? '');
$filterDateTo = trim($_GET['date_to'] ?? '');
$filterUsername = trim($_GET['username'] ?? '');
$filterId = trim($_GET['id'] ?? '');
$q = trim($_GET['q'] ?? '');

// Base query
if ($filterSubject > 0) {
    $sql = "SELECT DISTINCT s.*, g.name as grade_name
            FROM students s 
            JOIN grades g ON s.grade_id = g.id 
            JOIN student_subject ss ON s.id = ss.student_id
            WHERE s.academy_reference = " . $pdo->quote($currentSchool);
} else {
    $sql = "SELECT s.*, g.name as grade_name
            FROM students s 
            JOIN grades g ON s.grade_id = g.id
            WHERE s.academy_reference = " . $pdo->quote($currentSchool);
}

$conds = [];
$params = [];

if ($filterGrade > 0) { 
    $conds[] = 's.grade_id = ?'; 
    $params[] = $filterGrade; 
}
if ($filterSubject > 0) { 
    $conds[] = 'ss.subject_id = ?'; 
    $params[] = $filterSubject; 
}
if ($filterEmailStatus === 'verified') {
    $conds[] = 's.email_verified = TRUE'; 
} elseif ($filterEmailStatus === 'pending') {
    $conds[] = 's.email_verified = FALSE'; 
}
if ($filterDateFrom) { 
    $conds[] = 'DATE(s.created_at) >= ?'; 
    $params[] = $filterDateFrom; 
}
if ($filterDateTo) { 
    $conds[] = 'DATE(s.created_at) <= ?'; 
    $params[] = $filterDateTo; 
}
if ($filterUsername) { 
    $conds[] = 's.username LIKE ?'; 
    $params[] = '%'.$filterUsername.'%'; 
}
if ($filterId) { 
    $conds[] = 's.id = ?'; 
    $params[] = (int)$filterId; 
}
if ($q) { 
    $conds[] = '(s.full_name LIKE ? OR s.email LIKE ?)'; 
    $like = '%'.$q.'%'; 
    array_push($params, $like, $like); 
}

if (!empty($conds)) { 
    $sql .= ' AND ' . implode(' AND ', $conds); 
}

$sql .= ' ORDER BY s.created_at DESC';
$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$students = $stmt->fetchAll();

// Export based on format
if ($format === 'csv') {
    exportCSV($students, $schoolName);
} elseif ($format === 'pdf' || $format === 'html') {
    exportHTML($students, $schoolName, $schoolSettings, $schoolData, $currentSchool);
} else {
    // Default to CSV
    exportCSV($students, $schoolName);
}

// ============================================
// CSV Export
// ============================================
function exportCSV($students, $schoolName) {
    $filename = preg_replace('/[^a-zA-Z0-9_-]/', '_', $schoolName) . '_students_' . date('Y-m-d') . '.csv';
    
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Pragma: no-cache');
    header('Expires: 0');
    
    $output = fopen('php://output', 'w');
    
    // UTF-8 BOM for Excel
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // Header row
    fputcsv($output, [
        'ID',
        'Username',
        'Full Name',
        'Email',
        'Phone',
        'Grade',
        'Gender',
        'Date of Birth',
        'Guardian Name',
        'Guardian Phone',
        'MoMo Number',
        'Payment Status',
        'Email Verified',
        'Registered Date'
    ]);
    
    // Data rows
    foreach ($students as $student) {
        // Handle full_name or combine first_name + last_name
        $fullName = $student['full_name'] ?? '';
        if (empty($fullName) && isset($student['first_name']) && isset($student['last_name'])) {
            $fullName = trim($student['first_name'] . ' ' . $student['last_name']);
        }
        
        fputcsv($output, [
            $student['id'] ?? '',
            $student['username'] ?? '',
            $fullName,
            $student['email'] ?? '',
            $student['phone'] ?? '',
            $student['grade_name'] ?? '',
            $student['gender'] ?? '',
            $student['dob'] ?? '',
            $student['guardian_name'] ?? '',
            $student['guardian_phone'] ?? '',
            $student['momo_number'] ?? '',
            $student['payment_status'] ?? 'unpaid',
            isset($student['email_verified']) && $student['email_verified'] ? 'Yes' : 'No',
            isset($student['created_at']) ? date('Y-m-d H:i', strtotime($student['created_at'])) : ''
        ]);
    }
    
    fclose($output);
    exit;
}

// ============================================
// HTML Export with Professional Letterhead
// ============================================
function exportHTML($students, $schoolName, $schoolSettings, $schoolData, $currentSchool = null) {
    $logoUrl = $schoolSettings['logo_url'] ?? '../images/img-melanesys-logo.svg';
    $primaryColor = $schoolSettings['primary_color'] ?? '#4F46E5';
    $secondaryColor = $schoolSettings['secondary_color'] ?? '#10B981';
    
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>Student List - <?= htmlspecialchars($schoolName) ?></title>
        <style>
            @media print {
                .no-print { display: none !important; }
                @page { margin: 2cm; }
            }
            * {
                margin: 0;
                padding: 0;
                box-sizing: border-box;
            }
            body {
                font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                background: #fff;
                padding: 30px;
                color: #333;
            }
            /* Letterhead */
            .letterhead {
                border-bottom: 4px solid <?= $primaryColor ?>;
                padding-bottom: 25px;
                margin-bottom: 35px;
                display: flex;
                align-items: center;
                gap: 25px;
                background: linear-gradient(to right, #f8f9fa 0%, #ffffff 100%);
                padding: 20px;
                border-radius: 10px 10px 0 0;
            }
            .letterhead-logo {
                flex-shrink: 0;
            }
            .letterhead-logo img {
                height: 80px;
                width: auto;
                max-width: 120px;
                object-fit: contain;
            }
            .letterhead-text {
                flex-grow: 1;
            }
            .letterhead-text h1 {
                color: <?= $primaryColor ?>;
                font-size: 28px;
                margin-bottom: 8px;
                font-weight: 700;
            }
            .letterhead-text .contact-info {
                color: #666;
                font-size: 13px;
                line-height: 1.6;
            }
            .letterhead-text .contact-info strong {
                color: #333;
            }
            
            /* Report Header */
            .report-header {
                background: linear-gradient(135deg, <?= $primaryColor ?> 0%, <?= $secondaryColor ?> 100%);
                color: white;
                padding: 25px;
                border-radius: 10px;
                margin-bottom: 30px;
                text-align: center;
            }
            .report-header h2 {
                font-size: 24px;
                margin-bottom: 15px;
                font-weight: 700;
                letter-spacing: 1px;
            }
            .report-header .meta {
                display: flex;
                justify-content: center;
                gap: 30px;
                flex-wrap: wrap;
                font-size: 14px;
                opacity: 0.95;
            }
            .report-header .meta-item {
                display: flex;
                align-items: center;
                gap: 8px;
            }
            
            /* Action Buttons */
            .actions {
                text-align: center;
                margin-bottom: 25px;
            }
            .btn {
                display: inline-block;
                padding: 12px 28px;
                margin: 0 8px;
                background: <?= $primaryColor ?>;
                color: white;
                text-decoration: none;
                border-radius: 8px;
                font-weight: 600;
                font-size: 14px;
                border: none;
                cursor: pointer;
                transition: all 0.3s;
                box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            }
            .btn:hover {
                transform: translateY(-2px);
                box-shadow: 0 4px 12px rgba(0,0,0,0.2);
            }
            .btn-secondary {
                background: #6c757d;
            }
            
            /* Table */
            table {
                width: 100%;
                border-collapse: collapse;
                margin-top: 20px;
                font-size: 13px;
                box-shadow: 0 2px 8px rgba(0,0,0,0.08);
                border-radius: 8px;
                overflow: hidden;
            }
            thead {
                background: <?= $primaryColor ?>;
                color: white;
            }
            th {
                padding: 14px 10px;
                text-align: left;
                font-weight: 600;
                text-transform: uppercase;
                font-size: 11px;
                letter-spacing: 0.5px;
            }
            td {
                padding: 12px 10px;
                border-bottom: 1px solid #e9ecef;
            }
            tbody tr:nth-child(even) {
                background: #f8f9fa;
            }
            tbody tr:hover {
                background: #e9ecef;
            }
            
            /* Footer */
            .document-footer {
                margin-top: 50px;
                padding-top: 25px;
                border-top: 3px solid <?= $primaryColor ?>;
                text-align: center;
                color: #666;
                font-size: 12px;
            }
            .document-footer p {
                margin: 5px 0;
            }
            .document-footer .school-info {
                font-weight: 600;
                color: <?= $primaryColor ?>;
                font-size: 14px;
                margin-bottom: 8px;
            }
            
            /* No students message */
            .no-students {
                text-align: center;
                padding: 60px 20px;
                color: #999;
            }
            .no-students i {
                font-size: 48px;
                margin-bottom: 20px;
                display: block;
            }
        </style>
        <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    </head>
    <body>
        <!-- Letterhead -->
        <div class="letterhead">
            <div class="letterhead-logo">
                <img src="<?= htmlspecialchars($logoUrl) ?>" alt="<?= htmlspecialchars($schoolName) ?>" onerror="this.style.display='none'">
            </div>
            <div class="letterhead-text">
                <h1><?= htmlspecialchars($schoolName) ?></h1>
                <div class="contact-info">
                    <?php if (!empty($schoolSettings['school_address'])): ?>
                        <div><strong>📍 Address:</strong> <?= htmlspecialchars($schoolSettings['school_address']) ?></div>
                    <?php endif; ?>
                    <?php if (!empty($schoolSettings['school_phone'])): ?>
                        <div><strong>📞 Phone:</strong> <?= htmlspecialchars($schoolSettings['school_phone']) ?></div>
                    <?php endif; ?>
                    <?php if (!empty($schoolSettings['school_email'])): ?>
                        <div><strong>✉️ Email:</strong> <?= htmlspecialchars($schoolSettings['school_email']) ?></div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <!-- Report Header -->
        <div class="report-header">
            <h2>📋 STUDENT LIST REPORT</h2>
            <div class="meta">
                <div class="meta-item">
                    <i class="bi bi-calendar3"></i>
                    <strong>Generated:</strong> <?= date('l, F j, Y g:i A') ?>
                </div>
                <div class="meta-item">
                    <i class="bi bi-people"></i>
                    <strong>Total Students:</strong> <?= count($students) ?>
                </div>
                <?php if (!empty($schoolSettings['academic_year'])): ?>
                <div class="meta-item">
                    <i class="bi bi-book"></i>
                    <strong>Academic Year:</strong> <?= htmlspecialchars($schoolSettings['academic_year']) ?>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Action Buttons -->
        <div class="actions no-print">
            <button onclick="window.print()" class="btn">
                <i class="bi bi-printer me-2"></i>Print / Save as PDF
            </button>
            <a href="dashboard.php" class="btn btn-secondary">
                <i class="bi bi-arrow-left me-2"></i>Back to Dashboard
            </a>
        </div>
        
        <!-- Student Table -->
        <?php if (empty($students)): ?>
            <div class="no-students">
                <i class="bi bi-inbox"></i>
                <h3>No Students Found</h3>
                <p>Try adjusting your filters or add students to see data here.</p>
            </div>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Username</th>
                        <th>Full Name</th>
                        <th>Email</th>
                        <th>Phone</th>
                        <th>Grade</th>
                        <th>Guardian</th>
                        <th>Payment</th>
                        <th>Email Status</th>
                        <th>Registered</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($students as $student): 
                        // Handle full_name or combine first_name + last_name
                        $fullName = $student['full_name'] ?? '';
                        if (empty($fullName) && isset($student['first_name']) && isset($student['last_name'])) {
                            $fullName = trim($student['first_name'] . ' ' . $student['last_name']);
                        }
                    ?>
                    <tr>
                        <td><?= $student['id'] ?? '' ?></td>
                        <td><?= htmlspecialchars($student['username'] ?? '') ?></td>
                        <td><strong><?= htmlspecialchars($fullName) ?></strong></td>
                        <td><?= htmlspecialchars($student['email'] ?? 'N/A') ?></td>
                        <td><?= htmlspecialchars($student['phone'] ?? 'N/A') ?></td>
                        <td><?= htmlspecialchars($student['grade_name'] ?? '') ?></td>
                        <td><?= htmlspecialchars($student['guardian_name'] ?? 'N/A') ?></td>
                        <td>
                            <?php 
                            $paymentStatus = $student['payment_status'] ?? 'unpaid';
                            if ($paymentStatus === 'active' || $paymentStatus === 'paid'): ?>
                                <span style="color: #10b981; font-weight: 600;">✓ Active</span>
                            <?php elseif ($paymentStatus === 'pending'): ?>
                                <span style="color: #f59e0b;">⏳ Pending</span>
                            <?php else: ?>
                                <span style="color: #ef4444;">✗ Suspended</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if (isset($student['email_verified']) && $student['email_verified']): ?>
                                <span style="color: #10b981;">✓ Verified</span>
                            <?php else: ?>
                                <span style="color: #f59e0b;">⏳ Pending</span>
                            <?php endif; ?>
                        </td>
                        <td><?= isset($student['created_at']) ? date('M d, Y', strtotime($student['created_at'])) : 'N/A' ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
        
        <!-- Footer -->
        <div class="document-footer">
            <div class="school-info"><?= htmlspecialchars($schoolName) ?></div>
            <?php if (!empty($schoolSettings['school_email'])): ?>
                <p><?= htmlspecialchars($schoolSettings['school_email']) ?></p>
            <?php endif; ?>
            <p>Generated from School Management System | <?= date('Y-m-d H:i:s') ?></p>
            <?php if ($currentSchool): ?>
            <p style="color: #999; font-size: 11px; margin-top: 10px;">Reference: <?= htmlspecialchars($currentSchool) ?></p>
            <?php endif; ?>
        </div>
    </body>
    </html>
    <?php
    exit;
}
?>

