<?php
session_start();
require_once '../includes/functions.php';
require_once '../includes/school_context.php';
require_once '../includes/favicon_helper.php';
requireAdminLogin();
requireSchool(); // Ensure school is selected

// Get current school context
$currentSchool = getCurrentSchool();
$schoolName = getSchoolName();

$stats = getDashboardStats();

// Get database connection
$pdo = getDB();

// Get pending applications count
$stmt = $pdo->prepare("
    SELECT COUNT(*) 
    FROM student_applications 
    WHERE academy_reference = ? 
    AND application_status IN ('submitted', 'verified') 
    AND student_id IS NULL
");
$stmt->execute([$currentSchool]);
$pendingApplications = $stmt->fetchColumn();

// Filters
$filterGrade = (int)($_GET['grade'] ?? 0);
$filterSubject = (int)($_GET['subject'] ?? 0);
$filterEmailStatus = trim((string)($_GET['email_status'] ?? ''));
$filterDateFrom = trim((string)($_GET['date_from'] ?? ''));
$filterDateTo = trim((string)($_GET['date_to'] ?? ''));
$filterUsername = trim((string)($_GET['username'] ?? ''));
$filterId = trim((string)($_GET['id'] ?? ''));
$q = trim((string)($_GET['q'] ?? ''));

// Fetch grades for filter and modal
$grades = getGrades();

// Fetch students with optional filters
$pdo = getDB();

// Get school settings (for logo and favicon)
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM tenant_settings WHERE academy_reference = ?");
$stmt->execute([$currentSchool]);
$schoolSettings = [];
while ($row = $stmt->fetch()) {
    $schoolSettings[$row['setting_key']] = $row['setting_value'];
}
$schoolLogo = $schoolSettings['logo_url'] ?? '../images/img-melanesys-logo.svg';

// Determine filtering mode
$filteringModeSetting = $schoolSettings['filtering_mode'] ?? 'auto';
$totalStudentCount = $stats['total_students'];

if ($filteringModeSetting === 'auto') {
    // Auto-select based on student count
    if ($totalStudentCount < 2000) {
        $activeFilteringMode = 'client';
    } else {
        $activeFilteringMode = 'server';
    }
} else {
    $activeFilteringMode = $filteringModeSetting;
}

// Store for use in dashboard
$useClientSideFiltering = ($activeFilteringMode === 'client');

// Build the base query - ALWAYS filter by current school
if ($filterSubject > 0) {
    // If filtering by subject, we need to join with student_subject table
    $sql = "SELECT DISTINCT s.*, g.name as grade_name,
            CASE 
                WHEN EXISTS (SELECT 1 FROM payments p WHERE p.student_id = s.id AND p.status = 'completed') THEN 'Paid'
                WHEN EXISTS (SELECT 1 FROM payments p WHERE p.student_id = s.id AND p.status IN ('pending', 'failed')) THEN 'Pending'
                ELSE 'Unpaid'
            END as payment_status
            FROM students s 
            JOIN grades g ON s.grade_id = g.id 
            JOIN student_subject ss ON s.id = ss.student_id
            WHERE s.academy_reference = " . $pdo->quote($currentSchool);
} else {
$sql = "SELECT s.*, g.name as grade_name,
        CASE 
            WHEN EXISTS (SELECT 1 FROM payments p WHERE p.student_id = s.id AND p.status = 'completed') THEN 'Paid'
            WHEN EXISTS (SELECT 1 FROM payments p WHERE p.student_id = s.id AND p.status IN ('pending', 'failed')) THEN 'Pending'
            ELSE 'Unpaid'
        END as payment_status,
        s.moodle_status
        FROM students s JOIN grades g ON s.grade_id = g.id
        WHERE s.academy_reference = " . $pdo->quote($currentSchool);
}

$conds = [];
$params = [];

if ($filterGrade > 0) { 
    $conds[] = 's.grade_id = ?'; 
    $params[] = $filterGrade; 
}

if ($filterSubject > 0) { 
    $conds[] = 'ss.subject_id = ?'; 
    $params[] = $filterSubject; 
}

if ($filterEmailStatus !== '') { 
    if ($filterEmailStatus === 'verified') {
        $conds[] = 's.email_verified = TRUE'; 
    } elseif ($filterEmailStatus === 'pending') {
        $conds[] = 's.email_verified = FALSE'; 
    }
}

if ($filterDateFrom !== '') { 
    $conds[] = 'DATE(s.created_at) >= ?'; 
    $params[] = $filterDateFrom; 
}

if ($filterDateTo !== '') { 
    $conds[] = 'DATE(s.created_at) <= ?'; 
    $params[] = $filterDateTo; 
}

if ($filterUsername !== '') { 
    $conds[] = 's.username LIKE ?'; 
    $params[] = '%'.$filterUsername.'%'; 
}

if ($filterId !== '') { 
    $conds[] = 's.id = ?'; 
    $params[] = (int)$filterId; 
}

if ($q !== '') { 
    $conds[] = '(s.first_name LIKE ? OR s.last_name LIKE ? OR s.email LIKE ?)'; 
    $like = '%'.$q.'%'; 
    array_push($params, $like, $like, $like); 
}

if (!empty($conds)) { 
    $sql .= ' AND ' . implode(' AND ', $conds); 
}

// Pagination settings
$studentsPerPage = 20;
$currentPage = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$offset = ($currentPage - 1) * $studentsPerPage;

// Get total count for pagination (without LIMIT)
$countSql = $sql;
$countStmt = $pdo->prepare($countSql);
$countStmt->execute($params);
$totalStudents = count($countStmt->fetchAll());
$totalPages = ceil($totalStudents / $studentsPerPage);

// Add pagination to main query
$sql .= ' ORDER BY s.created_at DESC LIMIT ? OFFSET ?';
$params[] = $studentsPerPage;
$params[] = $offset;
$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$students = $stmt->fetchAll();

// Handle export_all request for printing/exporting all filtered data
if (isset($_GET['export_all']) && $_GET['export_all'] == '1') {
    // Remove pagination and get all filtered results
    $exportSql = str_replace(' ORDER BY s.created_at DESC LIMIT ? OFFSET ?', ' ORDER BY s.created_at DESC', $sql);
    $exportParams = array_slice($params, 0, -2); // Remove the last 2 parameters (limit and offset)
    
    $exportStmt = $pdo->prepare($exportSql);
    $exportStmt->execute($exportParams);
    $allStudents = $exportStmt->fetchAll();
    
    // Get subjects for each student
    foreach ($allStudents as &$student) {
        $student['subjects'] = getSubjectsForStudent($student['id']);
    }
    
    header('Content-Type: application/json');
    echo json_encode([
        'success' => true,
        'students' => $allStudents,
        'total' => count($allStudents)
    ]);
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($schoolName) ?> - Admin Dashboard</title>
    <?php echoFavicon($currentSchool); ?>
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
	<style>
		:root{ --brand-primary:#6f42c1; --brand-accent:#20c997; --brand-soft:#f3f0ff; }
		body{background:linear-gradient(135deg,var(--brand-soft),#e8fff9);} 
		.hero{background: radial-gradient(1000px 400px at 10% 0%, rgba(111,66,193,.15), rgba(32,201,151,.1)), linear-gradient(135deg, rgba(111,66,193,.12), rgba(32,201,151,.12)); border-radius: 1rem;}
		.brand-underline{position:relative; display:inline-block}
		.brand-underline::after{content:""; position:absolute; left:0; right:0; bottom:-.25rem; height:.35rem; background:linear-gradient(90deg,var(--brand-primary),var(--brand-accent)); border-radius:1rem}
		/* Modern Card Styling */
		.card-modern{
			border: 0; 
			box-shadow: 0 4px 20px rgba(0,0,0,0.08);
			border-radius: 1rem;
			overflow: hidden;
			transition: all 0.3s ease;
			background: #ffffff;
		}
		.card-modern:hover{
			transform: translateY(-2px);
			box-shadow: 0 8px 30px rgba(0,0,0,0.12);
		}
		.card-modern .card-header{
			background: linear-gradient(135deg, #f8f9fa, #ffffff);
			border-bottom: 1px solid rgba(0,0,0,0.05);
			padding: 1.5rem;
		}
		.card-modern .card-body{
			padding: 1.5rem;
		}
		
		/* Professional Stat Cards */
		.stat-card{
			color: #fff; 
			border: 0;
			border-radius: 1rem;
			overflow: hidden;
			position: relative;
			transition: all 0.3s ease;
			min-height: 140px;
		}
		.stat-card:hover{
			transform: translateY(-3px);
			box-shadow: 0 12px 40px rgba(0,0,0,0.15);
		}
		.stat-card::before{
			content: '';
			position: absolute;
			top: 0;
			left: 0;
			right: 0;
			bottom: 0;
			background: linear-gradient(135deg, rgba(255,255,255,0.1), rgba(255,255,255,0.05));
			pointer-events: none;
		}
		.stat-card .card-body{
			position: relative;
			z-index: 1;
			padding: 1.5rem;
			height: 100%;
			display: flex;
			flex-direction: column;
			justify-content: center;
		}
		.stat-icon{
			font-size: 2.5rem; 
			opacity: 0.9;
			filter: drop-shadow(0 2px 4px rgba(0,0,0,0.1));
		}
		
		/* Enhanced Brand Gradients */
		.bg-brand-1{
			background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
			box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
		}
		.bg-brand-2{
			background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
			box-shadow: 0 8px 25px rgba(17, 153, 142, 0.3);
		}
		.bg-brand-3{
			background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
			box-shadow: 0 8px 25px rgba(240, 147, 251, 0.3);
		}
		.bg-brand-4{
			background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
			box-shadow: 0 8px 25px rgba(79, 172, 254, 0.3);
		}
		
		/* Modern Buttons */
		.btn-gradient{
			background: linear-gradient(135deg, var(--brand-primary), var(--brand-accent));
			border: 0;
			border-radius: 0.75rem;
			padding: 0.75rem 1.5rem;
			font-weight: 600;
			transition: all 0.3s ease;
			box-shadow: 0 4px 15px rgba(111, 66, 193, 0.3);
		}
		.btn-gradient:hover{
			transform: translateY(-2px);
			box-shadow: 0 8px 25px rgba(111, 66, 193, 0.4);
			filter: brightness(1.05);
		}
		
		/* Checkbox Styles */
		.form-check-input:checked{background-color:var(--brand-primary); border-color:var(--brand-primary)}
		.form-check-label{cursor:pointer; user-select:none}
		.subject-checkbox:hover{background-color:rgba(111,66,193,.05); border-radius:.375rem}
		
		/* Modern Form Enhancements for Admin */
		.form-section {
			background: rgba(248, 249, 250, 0.5);
			border-radius: 12px;
			padding: 1.25rem;
			border: 1px solid rgba(0, 0, 0, 0.05);
			margin-bottom: 1rem;
		}
		.section-title {
			font-size: 0.9rem;
			font-weight: 600;
			letter-spacing: 0.5px;
			text-transform: uppercase;
			border-bottom: 2px solid #e9ecef;
			padding-bottom: 0.5rem;
			margin-bottom: 1rem;
		}
		.form-control-lg {
			border-radius: 8px;
			border: 2px solid #e9ecef;
			transition: all 0.3s ease;
		}
		.form-control-lg:focus {
			border-color: var(--brand-primary);
			box-shadow: 0 0 0 0.2rem rgba(111, 66, 193, 0.25);
		}
		.form-select-lg {
			border-radius: 8px;
			border: 2px solid #e9ecef;
			transition: all 0.3s ease;
		}
		.form-select-lg:focus {
			border-color: var(--brand-primary);
			box-shadow: 0 0 0 0.2rem rgba(111, 66, 193, 0.25);
		}
		.form-label {
			color: #495057;
			margin-bottom: 0.5rem;
		}
		.form-text {
			font-size: 0.8rem;
			color: #6c757d;
		}
		.alert-warning {
			background: rgba(255, 193, 7, 0.1);
			border: 1px solid rgba(255, 193, 7, 0.3);
			border-radius: 8px;
		}
		#modalSubjectsContainer {
			background: linear-gradient(135deg, #f8f9fa, #ffffff);
			border: 2px dashed #dee2e6;
			transition: all 0.3s ease;
		}
		#modalSubjectsContainer:hover {
			border-color: var(--brand-primary);
		}
		.modal-body {
			padding: 1.5rem;
		}
		.modal-footer {
			background: rgba(111, 66, 193, 0.05);
			border-top: 1px solid rgba(111, 66, 193, 0.1);
		}
		
		/* Enhanced Badges */
		.badge-soft{
			background: rgba(255,255,255,0.2); 
			color: #fff;
			border-radius: 0.5rem;
			padding: 0.5rem 1rem;
			font-weight: 500;
			backdrop-filter: blur(10px);
		}
		.filter-pill{border:0; background:#f8f9fa}
		/* Modal styling to match registration */
		.modal-header.gradient{background:linear-gradient(90deg,var(--brand-primary),var(--brand-accent)); color:#fff}
		.input-icon{position:relative}
		.input-icon .bi{position:absolute; left:.9rem; top:50%; transform:translateY(-50%); color:#6c757d}
		.input-icon input, .input-icon select{padding-left:2.4rem}
		/* Professional Filter Section Styling */
		.filter-section{
			background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
			border: 1px solid rgba(111,66,193,0.08);
			border-radius: 1rem;
			padding: 2rem;
			margin-bottom: 2rem;
			box-shadow: 0 4px 20px rgba(0,0,0,0.08);
			position: relative;
			overflow: hidden;
			transition: all 0.3s ease;
		}
		.filter-section:hover{
			box-shadow: 0 8px 30px rgba(0,0,0,0.12);
		}
		.filter-section::before{
			content: '';
			position: absolute;
			top: 0;
			left: 0;
			right: 0;
			height: 4px;
			background: linear-gradient(90deg, var(--brand-primary), var(--brand-accent));
		}
		.filter-header{
			display: flex;
			justify-content: space-between;
			align-items: center;
			margin-bottom: 2rem;
			padding-bottom: 1.5rem;
			border-bottom: 2px solid rgba(111,66,193,0.1);
		}
		.filter-title{
			font-size: 1.3rem;
			font-weight: 700;
			color: #2c3e50;
			margin: 0;
			display: flex;
			align-items: center;
			gap: 0.75rem;
			letter-spacing: -0.01em;
		}
		.filter-title i{
			color: var(--brand-primary);
			font-size: 1.4rem;
		}
		.auto-filter-badge{
			background: linear-gradient(135deg, #20c997, #17a2b8);
			color: white;
			padding: 0.5rem 1rem;
			border-radius: 1.5rem;
			font-size: 0.8rem;
			font-weight: 600;
			display: flex;
			align-items: center;
			gap: 0.5rem;
			box-shadow: 0 4px 15px rgba(32,201,151,0.3);
			text-transform: uppercase;
			letter-spacing: 0.5px;
		}
		.filter-grid{
			display: grid;
			grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
			gap: 1.5rem;
			align-items: end;
		}
		.filter-group{
			display: flex;
			flex-direction: column;
			gap: 0.75rem;
		}
		.filter-label{
			font-size: 0.85rem;
			font-weight: 700;
			color: #495057;
			margin: 0;
			text-transform: uppercase;
			letter-spacing: 0.8px;
		}
		.filter-input{
			border: 2px solid rgba(111,66,193,0.1);
			border-radius: 0.75rem;
			padding: 0.875rem 1rem;
			font-size: 0.9rem;
			transition: all 0.3s ease;
			background: #ffffff;
			box-shadow: 0 2px 8px rgba(0,0,0,0.04);
			font-weight: 500;
		}
		.filter-input:focus{
			border-color: var(--brand-primary);
			box-shadow: 0 0 0 0.2rem rgba(111,66,193,0.15), 0 4px 20px rgba(111,66,193,0.15);
			background: #ffffff;
			transform: translateY(-2px);
		}
		.filter-input:hover{
			border-color: rgba(111,66,193,0.3);
			box-shadow: 0 4px 15px rgba(0,0,0,0.08);
			transform: translateY(-1px);
		}
		.filter-input::placeholder{
			color: #adb5bd;
			font-weight: 400;
		}
		.input-group .filter-input{
			border-left: none;
			border-radius: 0 0.75rem 0.75rem 0;
		}
		.input-group .input-group-text{
			background: linear-gradient(135deg, rgba(111,66,193,0.05), rgba(111,66,193,0.1));
			border: 2px solid rgba(111,66,193,0.1);
			border-right: none;
			border-radius: 0.75rem 0 0 0.75rem;
			color: var(--brand-primary);
			font-weight: 700;
			font-size: 0.9rem;
		}
		.input-group:focus-within .input-group-text{
			border-color: var(--brand-primary);
			background: linear-gradient(135deg, rgba(111,66,193,0.1), rgba(111,66,193,0.15));
		}
		.filter-actions{
			display: flex;
			gap: 1rem;
			align-items: center;
			padding-top: 1.5rem;
			border-top: 2px solid rgba(111,66,193,0.1);
			margin-top: 1.5rem;
		}
		.clear-btn{
			background: linear-gradient(135deg, #6c757d, #495057);
			border: none;
			color: white;
			padding: 0.875rem 2rem;
			border-radius: 0.75rem;
			font-size: 0.9rem;
			font-weight: 600;
			transition: all 0.3s ease;
			box-shadow: 0 4px 15px rgba(108,117,125,0.3);
			text-transform: uppercase;
			letter-spacing: 0.5px;
		}
		.clear-btn:hover{
			transform: translateY(-2px);
			box-shadow: 0 8px 25px rgba(108,117,125,0.4);
			color: white;
		}
		/* Loading and transition effects */
		.filter-loading{
			opacity: 0.7;
			pointer-events: none;
			transition: all 0.3s ease;
		}
		.filter-section.loading{
			position: relative;
		}
		.filter-section.loading::after{
			content: '';
			position: absolute;
			top: 0;
			left: 0;
			right: 0;
			bottom: 0;
			background: rgba(255,255,255,0.8);
			display: flex;
			align-items: center;
			justify-content: center;
			border-radius: 1rem;
		}
		@media (max-width: 768px) {
			.filter-grid{
				grid-template-columns: 1fr;
				gap: 1rem;
			}
			.filter-header{
				flex-direction: column;
				gap: 1rem;
				align-items: flex-start;
			}
			.filter-actions{
				width: 100%;
				justify-content: center;
			}
		}
		/* Compact Subjects Styling */
		.subjects-container{
			display: flex;
			flex-wrap: wrap;
			gap: 0.1rem;
			align-items: center;
			max-width: 300px;
		}
		.subjects-container .badge{
			font-size: 0.6rem;
			padding: 0.15rem 0.3rem;
			border-radius: 0.25rem;
			font-weight: 500;
			line-height: 1.2;
		}
		
		/* Enhanced Visual Hierarchy */
		.hero h1{
			font-weight: 700;
			letter-spacing: -0.02em;
		}
		.hero p{
			font-size: 1.1rem;
			color: #6c757d;
		}
		
		/* Modern Table Enhancements */
		.table th{
			font-weight: 600;
			text-transform: uppercase;
			font-size: 0.8rem;
			letter-spacing: 0.5px;
			color: #495057;
			border-bottom: 2px solid #e9ecef;
		}
		.table td{
			vertical-align: middle;
			padding: 1rem 0.75rem;
		}
		
		/* Enhanced Export Button */
		.btn-outline-primary{
			border: 2px solid var(--brand-primary);
			color: var(--brand-primary);
			font-weight: 500;
			border-radius: 0.75rem;
			transition: all 0.3s ease;
		}
		.btn-outline-primary:hover{
			background: var(--brand-primary);
			border-color: var(--brand-primary);
			transform: translateY(-1px);
			box-shadow: 0 4px 15px rgba(111, 66, 193, 0.3);
		}
		
		/* Modern Dropdown */
		.dropdown-menu{
			border: 0;
			box-shadow: 0 8px 30px rgba(0,0,0,0.12);
			border-radius: 0.75rem;
			padding: 0.5rem;
		}
		.dropdown-item{
			border-radius: 0.5rem;
			margin: 0.125rem 0;
			font-weight: 500;
			transition: all 0.2s ease;
		}
		.dropdown-item:hover{
			background: linear-gradient(135deg, rgba(111,66,193,0.1), rgba(32,201,151,0.1));
			transform: translateX(4px);
		}
		
		/* Professional Chart Cards */
		.chart-card{
			height: 400px;
		}
		.chart-card .card-body{
			height: 320px;
			display: flex;
			align-items: center;
			justify-content: center;
		}
		.chart-card canvas{
			max-height: 100%;
			max-width: 100%;
		}
		
		/* Consistent Card Heights */
		.row.g-4 .card{
			display: flex;
			flex-direction: column;
		}
		.row.g-4 .card .card-body{
			flex: 1;
		}
		
		/* Professional Active Filters Display */
		.active-filters-display{
			background: linear-gradient(135deg, rgba(111,66,193,0.05), rgba(32,201,151,0.05));
			border: 1px solid rgba(111,66,193,0.1);
			border-radius: 0.75rem;
			padding: 1rem 1.5rem;
			margin-bottom: 1.5rem;
			box-shadow: 0 2px 8px rgba(0,0,0,0.04);
		}
		.active-filters-display i{
			color: var(--brand-primary);
			font-size: 1.1rem;
		}
		.active-filters-display strong{
			color: #2c3e50;
			font-weight: 700;
		}
		.active-filters-display span{
			color: #495057;
			font-weight: 500;
		}
		
		/* Professional Pagination Styling */
		.pagination-info{
			font-size: 0.9rem;
			color: #6c757d;
			font-weight: 500;
		}
		.pagination{
			--bs-pagination-color: var(--brand-primary);
			--bs-pagination-bg: #ffffff;
			--bs-pagination-border-color: rgba(111,66,193,0.1);
			--bs-pagination-hover-color: #ffffff;
			--bs-pagination-hover-bg: var(--brand-primary);
			--bs-pagination-hover-border-color: var(--brand-primary);
			--bs-pagination-focus-color: #ffffff;
			--bs-pagination-focus-bg: var(--brand-primary);
			--bs-pagination-focus-border-color: var(--brand-primary);
			--bs-pagination-active-color: #ffffff;
			--bs-pagination-active-bg: var(--brand-primary);
			--bs-pagination-active-border-color: var(--brand-primary);
			--bs-pagination-disabled-color: #6c757d;
			--bs-pagination-disabled-bg: #ffffff;
			--bs-pagination-disabled-border-color: #dee2e6;
		}
		.page-link{
			border-radius: 0.5rem;
			margin: 0 0.125rem;
			border: 2px solid rgba(111,66,193,0.1);
			font-weight: 500;
			transition: all 0.3s ease;
			box-shadow: 0 2px 4px rgba(0,0,0,0.04);
		}
		.page-link:hover{
			transform: translateY(-1px);
			box-shadow: 0 4px 12px rgba(111,66,193,0.2);
		}
		.page-item.active .page-link{
			background: linear-gradient(135deg, var(--brand-primary), var(--brand-accent));
			border-color: var(--brand-primary);
			box-shadow: 0 4px 15px rgba(111,66,193,0.3);
		}
		.page-item.disabled .page-link{
			opacity: 0.5;
			cursor: not-allowed;
		}
		
		/* Professional Action Buttons */
		.action-btn{
			border-radius: 0.5rem;
			padding: 0.5rem 0.75rem;
			font-size: 0.875rem;
			font-weight: 500;
			transition: all 0.3s ease;
			border: 2px solid;
			position: relative;
			overflow: hidden;
		}
		.action-btn::before{
			content: '';
			position: absolute;
			top: 0;
			left: -100%;
			width: 100%;
			height: 100%;
			background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
			transition: left 0.5s;
		}
		.action-btn:hover::before{
			left: 100%;
		}
		.action-btn:hover{
			transform: translateY(-2px);
			box-shadow: 0 4px 12px rgba(0,0,0,0.15);
		}
		.btn-outline-primary.action-btn{
			border-color: var(--brand-primary);
			color: var(--brand-primary);
		}
		.btn-outline-primary.action-btn:hover{
			background: var(--brand-primary);
			border-color: var(--brand-primary);
			color: white;
			box-shadow: 0 4px 15px rgba(111,66,193,0.3);
		}
		.btn-outline-danger.action-btn{
			border-color: #dc3545;
			color: #dc3545;
		}
		.btn-outline-danger.action-btn:hover{
			background: #dc3545;
			border-color: #dc3545;
			color: white;
			box-shadow: 0 4px 15px rgba(220,53,69,0.3);
		}
		.btn-group .action-btn{
			border-radius: 0;
		}
		.btn-group .action-btn:first-child{
			border-top-left-radius: 0.5rem;
			border-bottom-left-radius: 0.5rem;
		}
		.btn-group .action-btn:last-child{
			border-top-right-radius: 0.5rem;
			border-bottom-right-radius: 0.5rem;
		}
	</style>
</head>
<body class="bg-light">
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container-fluid">
            <a class="navbar-brand d-flex align-items-center" href="dashboard.php">
                <img src="<?= htmlspecialchars($schoolLogo) ?>" alt="<?= htmlspecialchars($schoolName) ?>" 
                     style="height: 32px; width: auto; max-width: 40px; object-fit: contain; margin-right: 10px;"
                     onerror="this.src='../images/img-melanesys-logo.svg'">
                <span>Admin Dashboard</span>
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="restore_student.php">
                    <i class="bi bi-arrow-clockwise me-1"></i>Restore Students
                </a>
                <a class="nav-link" href="configure_subjects.php">
                    <i class="bi bi-gear me-1"></i>Configure Subjects
                </a>
                <a class="nav-link" href="moodle/configure_mapping.php">
                    <i class="bi bi-mortarboard me-1"></i>Moodle Setup
                </a>
                <a class="nav-link" href="storage/usage.php">
                    <i class="bi bi-hdd-stack me-1"></i>Storage
                </a>
                <a class="nav-link" href="settings/manage_student_payments.php">
                    <i class="bi bi-credit-card me-1"></i>Payments
                </a>
                <a class="nav-link" href="settings.php">
                    <i class="bi bi-sliders me-1"></i>Settings
                </a>
                <a class="nav-link" href="logout.php">
                    <i class="bi bi-box-arrow-right me-1"></i>Logout
                </a>
            </div>
        </div>
    </nav>

    <!-- School Context Banner -->
    <?php if (isSuperAdmin()): ?>
    <div class="alert alert-warning mb-0 rounded-0 border-0 shadow-sm" style="border-left: 4px solid #f59e0b !important;">
        <div class="container-fluid">
            <div class="d-flex align-items-center justify-content-between">
                <div class="d-flex align-items-center gap-3">
                    <i class="bi bi-shield-check fs-4"></i>
                    <div>
                        <strong class="d-block">Super Admin Mode</strong>
                        <small>Viewing: <span class="badge bg-dark"><?= htmlspecialchars($schoolName ?? $currentSchool) ?></span> | 
                        Logged in as: <strong><?= htmlspecialchars($_SESSION['admin_username'] ?? 'Super Admin') ?></strong></small>
                    </div>
                </div>
                <div class="d-flex gap-2">
                    <a href="../super_admin/dashboard.php" class="btn btn-sm btn-outline-dark">
                        <i class="bi bi-arrow-left me-1"></i>Back to Super Admin
                    </a>
                    <a href="../super_admin/schools/view.php?id=<?= $_SESSION['academy_id'] ?? '' ?>" class="btn btn-sm btn-dark">
                        <i class="bi bi-building me-1"></i>School Details
                    </a>
                </div>
            </div>
        </div>
    </div>
    <?php else: ?>
    <div class="alert alert-info mb-0 rounded-0 border-0" style="border-left: 4px solid #3b82f6 !important;">
        <div class="container-fluid">
            <div class="d-flex align-items-center justify-content-between">
                <div class="d-flex align-items-center gap-3">
                    <img src="<?= htmlspecialchars($schoolLogo) ?>" 
                         alt="<?= htmlspecialchars($schoolName) ?>" 
                         style="height: 48px; width: auto; max-width: 60px; object-fit: contain; border-radius: 8px; background: white; padding: 4px;"
                         onerror="this.outerHTML='<i class=\'bi bi-building fs-4\'></i>'">
                    <div>
                        <strong class="d-block"><?= htmlspecialchars($schoolName ?? 'Your School') ?></strong>
                        <small>Logged in as: <strong><?= htmlspecialchars($_SESSION['admin_username'] ?? 'Administrator') ?></strong> | 
                        School Code: <span class="badge bg-primary"><?= htmlspecialchars($currentSchool) ?></span></small>
                    </div>
                </div>
                <div>
                    <small class="text-muted">
                        <i class="bi bi-clock me-1"></i><?= date('l, F j, Y') ?>
                    </small>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <div class="container-fluid my-4">
		<div class="hero p-4 p-md-5 mb-4 text-dark">
			<div class="d-flex align-items-center mb-2">
				<span class="badge bg-dark-subtle text-dark me-2">Overview</span>
				<small class="text-muted">Metrics, trends, and student management</small>
				<?php if (isSuperAdmin()): ?>
					<span class="badge bg-info text-white ms-2" title="Filtering Mode">
						<i class="bi bi-funnel me-1"></i>
						<?= ucfirst($activeFilteringMode) ?>-side Filtering
						<?php if ($filteringModeSetting === 'auto'): ?>
							<small>(Auto)</small>
						<?php endif; ?>
					</span>
				<?php endif; ?>
			</div>
			<div class="d-flex flex-wrap justify-content-between align-items-end gap-3">
				<div>
					<h1 class="display-6 fw-bold mb-1 brand-underline">Welcome, <?= htmlspecialchars($schoolName ?? 'Admin') ?></h1>
					<p class="mb-0">Monitor registrations, manage students, and export data for <?= htmlspecialchars($currentSchool) ?>.</p>
				</div>
				<div class="d-flex gap-2">
					<button class="btn btn-gradient text-white" data-bs-toggle="modal" data-bs-target="#createStudentModal"><i class="bi bi-person-plus me-1"></i>New Student</button>
					<a class="btn btn-outline-warning" href="restore_student.php"><i class="bi bi-arrow-clockwise me-1"></i>Restore Students</a>
				</div>
            </div>
        </div>

        <!-- Stats Cards -->
        <div class="row g-4 mb-4">
            <div class="col-md-3">
				<div class="card stat-card bg-brand-1" style="height: 140px;">
                    <div class="card-body d-flex flex-column justify-content-center">
						<div class="d-flex justify-content-between align-items-center">
                            <div>
								<div class="small opacity-75 mb-1">Total Students</div>
								<div class="h3 mb-0"><?php echo $stats['total_students']; ?></div>
                            </div>
							<i class="bi bi-people-fill stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
				<div class="card stat-card bg-brand-2" style="height: 140px;">
                    <div class="card-body d-flex flex-column justify-content-center">
						<div class="d-flex justify-content-between align-items-center">
                            <div>
								<div class="small opacity-75 mb-1">Recent 7 days</div>
								<div class="h3 mb-0"><?php echo $stats['recent_registrations']; ?></div>
                            </div>
							<i class="bi bi-graph-up stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
				<div class="card stat-card bg-brand-3" style="height: 140px;">
                    <div class="card-body d-flex flex-column justify-content-center">
						<div class="d-flex justify-content-between align-items-center">
                            <div>
								<div class="small opacity-75 mb-1">Remaining Spots</div>
								<?php 
								$maxStudents = (int)($schoolSettings['max_students'] ?? 500);
								$remainingSpots = max(0, $maxStudents - $stats['total_students']);
								?>
								<div class="h3 mb-0"><?php echo $remainingSpots; ?></div>
								<span class="badge badge-soft rounded-pill mt-1">Max <?php echo $maxStudents; ?></span>
                            </div>
							<i class="bi bi-person-plus stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
				<a href="applications/list.php" class="text-decoration-none">
					<div class="card stat-card bg-brand-4" style="height: 140px;">
						<div class="card-body d-flex flex-column justify-content-center">
							<div class="d-flex justify-content-between align-items-center">
								<div>
									<div class="small opacity-75 mb-1">Pending Applications</div>
									<div class="h3 mb-0"><?php echo $pendingApplications; ?></div>
									<?php if ($pendingApplications > 0): ?>
									<span class="badge bg-light text-dark rounded-pill mt-1">
										<i class="bi bi-exclamation-circle me-1"></i>Needs Review
									</span>
									<?php else: ?>
									<span class="badge bg-light text-dark rounded-pill mt-1">
										<i class="bi bi-check-circle me-1"></i>All Clear
									</span>
									<?php endif; ?>
								</div>
								<i class="bi bi-file-earmark-text stat-icon"></i>
							</div>
						</div>
					</div>
				</a>
            </div>
        </div>

        <!-- Email Verification Stats -->
        <div class="row g-4 mb-4">
            <div class="col-md-6">
				<div class="card stat-card" style="background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%); box-shadow: 0 8px 25px rgba(17, 153, 142, 0.3); height: 140px;">
                    <div class="card-body d-flex flex-column justify-content-center">
						<div class="d-flex justify-content-between align-items-center">
                            <div>
								<div class="small opacity-75 mb-1" style="font-weight: 500; letter-spacing: 0.5px;">Verified Emails</div>
								<div class="h3 mb-0" style="font-weight: 700; text-shadow: 0 2px 4px rgba(0,0,0,0.1);"><?php echo $stats['verified_emails']; ?></div>
                            </div>
							<i class="bi bi-envelope-check stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
				<div class="card stat-card" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); box-shadow: 0 8px 25px rgba(240, 147, 251, 0.3); height: 140px;">
                    <div class="card-body d-flex flex-column justify-content-center">
						<div class="d-flex justify-content-between align-items-center">
                            <div>
								<div class="small opacity-75 mb-1" style="font-weight: 500; letter-spacing: 0.5px;">Pending Verification</div>
								<div class="h3 mb-0" style="font-weight: 700; text-shadow: 0 2px 4px rgba(0,0,0,0.1);"><?php echo $stats['pending_verifications']; ?></div>
                            </div>
							<i class="bi bi-envelope-exclamation stat-icon"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Wallet Balance Section -->
        <?php
        require_once '../includes/wallet_functions.php';
        $admin_id = $_SESSION['admin_id'] ?? 1;
        $wallet = getAdminWallet($admin_id);
        $pending_withdrawals = getPendingWithdrawals($admin_id);
        ?>
        <div class="row g-4 mb-4">
            <!-- Wallet Balance Card -->
            <div class="col-md-4">
                <?php
                // Get payment breakdown by method - filtered by current school
                $payment_breakdown_stmt = $pdo->prepare("
                    SELECT 
                        payment_method,
                        COUNT(*) as count,
                        SUM(amount) as total_amount,
                        SUM(admin_earnings) as total_earnings
                    FROM payments 
                    WHERE status = 'completed' AND academy_reference = ?
                    GROUP BY payment_method
                ");
                $payment_breakdown_stmt->execute([$currentSchool]);
                $payment_breakdown = $payment_breakdown_stmt->fetchAll(PDO::FETCH_ASSOC);
                
                $momo_payments = 0;
                $card_payments = 0;
                $momo_earnings = 0;
                $card_earnings = 0;
                
                foreach ($payment_breakdown as $method) {
                    if ($method['payment_method'] === 'MOMO') {
                        $momo_payments = $method['count'];
                        $momo_earnings = $method['total_earnings'] ?? 0;
                    } elseif ($method['payment_method'] === 'CARD') {
                        $card_payments = $method['count'];
                        $card_earnings = $method['total_earnings'] ?? 0;
                    }
                }
                ?>
                <div class="card stat-card" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4); min-height: 200px;">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-start mb-3">
                            <div>
                                <div class="small opacity-75 mb-1" style="font-weight: 500; letter-spacing: 0.5px; color: #fff;">Your Wallet Balance</div>
                                <div class="h2 mb-0" style="font-weight: 700; text-shadow: 0 2px 4px rgba(0,0,0,0.2); color: #fff;">
                                    SZL <?php echo number_format($wallet['balance'] ?? 0, 2); ?>
                                </div>
                            </div>
                            <i class="bi bi-wallet2" style="font-size: 2.5rem; opacity: 0.3; color: #fff;"></i>
                        </div>
                        <div class="mt-3 pt-3" style="border-top: 1px solid rgba(255,255,255,0.2);">
                            <div class="d-flex justify-content-between text-white-50 small">
                                <span><i class="bi bi-phone me-1"></i> Mobile Money:</span>
                                <span class="text-white fw-bold"><?php echo $momo_payments; ?> (SZL <?php echo number_format($momo_earnings, 2); ?>)</span>
                            </div>
                            <div class="d-flex justify-content-between text-white-50 small mt-1">
                                <span><i class="bi bi-credit-card me-1"></i> Card Payments:</span>
                                <span class="text-white fw-bold"><?php echo $card_payments; ?> (SZL <?php echo number_format($card_earnings, 2); ?>)</span>
                            </div>
                            <div class="d-flex justify-content-between text-white-50 small mt-2 pt-2" style="border-top: 1px solid rgba(255,255,255,0.15);">
                                <span>Total Withdrawn:</span>
                                <span class="text-white fw-bold">SZL <?php echo number_format($wallet['total_withdrawn'] ?? 0, 2); ?></span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Platform Fees (Your Earnings) -->
            <div class="col-md-4">
                <?php
                // Get platform fees summary - filtered by current school
                $fees_summary_stmt = $pdo->prepare("
                    SELECT 
                        COUNT(*) as total_transactions,
                        SUM(fee_amount) as total_platform_fees,
                        SUM(gross_amount) as total_revenue
                    FROM platform_fees 
                    WHERE admin_id = ? AND academy_reference = ?
                ");
                $fees_summary_stmt->execute([$admin_id, $currentSchool]);
                $platform_fees_summary = $fees_summary_stmt->fetch();
                
                // Get fees breakdown by payment method - filtered by current school
                $fees_method_stmt = $pdo->prepare("
                    SELECT 
                        payment_method,
                        COUNT(*) as count,
                        SUM(fee_amount) as total_fees
                    FROM platform_fees 
                    WHERE admin_id = ? AND academy_reference = ?
                    GROUP BY payment_method
                ");
                $fees_method_stmt->execute([$admin_id, $currentSchool]);
                $fees_by_method = $fees_method_stmt->fetchAll(PDO::FETCH_ASSOC);
                
                $momo_fee_count = 0;
                $card_fee_count = 0;
                $momo_fees = 0;
                $card_fees = 0;
                
                foreach ($fees_by_method as $method) {
                    if ($method['payment_method'] === 'MOMO') {
                        $momo_fee_count = $method['count'];
                        $momo_fees = $method['total_fees'] ?? 0;
                    } elseif ($method['payment_method'] === 'CARD') {
                        $card_fee_count = $method['count'];
                        $card_fees = $method['total_fees'] ?? 0;
                    }
                }
                ?>
                <div class="card stat-card" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); box-shadow: 0 8px 25px rgba(240, 147, 251, 0.3); min-height: 200px;">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-start mb-3">
                            <div>
                                <div class="small opacity-75 mb-1" style="font-weight: 500; letter-spacing: 0.5px; color: #fff;">Platform Fees Earned</div>
                                <div class="h2 mb-0" style="font-weight: 700; text-shadow: 0 2px 4px rgba(0,0,0,0.2); color: #fff;">
                                    SZL <?php echo number_format($platform_fees_summary['total_platform_fees'] ?? 0, 2); ?>
                                </div>
                            </div>
                            <i class="bi bi-currency-dollar" style="font-size: 2.5rem; opacity: 0.3; color: #fff;"></i>
                        </div>
                        <div class="mt-3 pt-3" style="border-top: 1px solid rgba(255,255,255,0.2);">
                            <div class="d-flex justify-content-between text-white-50 small">
                                <span><i class="bi bi-phone me-1"></i> MoMo Fees:</span>
                                <span class="text-white fw-bold"><?php echo $momo_fee_count; ?> (SZL <?php echo number_format($momo_fees, 2); ?>)</span>
                            </div>
                            <div class="d-flex justify-content-between text-white-50 small mt-1">
                                <span><i class="bi bi-credit-card me-1"></i> Card Fees:</span>
                                <span class="text-white fw-bold"><?php echo $card_fee_count; ?> (SZL <?php echo number_format($card_fees, 2); ?>)</span>
                            </div>
                            <div class="d-flex justify-content-between text-white-50 small mt-2 pt-2" style="border-top: 1px solid rgba(255,255,255,0.15);">
                                <span>Total Revenue:</span>
                                <span class="text-white fw-bold">SZL <?php echo number_format($platform_fees_summary['total_revenue'] ?? 0, 2); ?></span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Pending Withdrawals Alert -->
        <?php if (!empty($pending_withdrawals)): ?>
        <div class="row g-4 mb-4">
            <div class="col-12">
                <div class="alert alert-info d-flex align-items-center" role="alert">
                    <i class="bi bi-info-circle-fill me-2"></i>
                    <div class="flex-grow-1">
                        <strong>Pending Withdrawal:</strong> You have <?php echo count($pending_withdrawals); ?> withdrawal request(s) being processed.
                        <?php foreach ($pending_withdrawals as $withdrawal): ?>
                            <span class="badge bg-info ms-2">
                                SZL <?php echo number_format($withdrawal['amount'], 2); ?> - <?php echo ucfirst(strtolower($withdrawal['status'])); ?>
                            </span>
                        <?php endforeach; ?>
                    </div>
                    <a href="withdrawals.php" class="btn btn-info btn-sm">View Details</a>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Bot Monitoring Section -->
        <div class="row g-4 mb-4">
            <div class="col-12">
                <div class="card card-modern">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0" style="color: #2c3e50; font-weight: 600;">
                            <i class="bi bi-robot me-2" style="color: var(--brand-primary);"></i>Bot Monitoring
                        </h5>
                        <div class="d-flex gap-2">
                            <button type="button" class="btn btn-sm btn-outline-primary" onclick="refreshBotStatus()">
                                <i class="bi bi-arrow-clockwise me-1"></i>Refresh
                            </button>
                            <button type="button" class="btn btn-sm btn-success" onclick="triggerBot()" id="triggerBotBtn">
                                <i class="bi bi-play-circle me-1"></i>Run Bot Now
                            </button>
                        </div>
                    </div>
                    <div class="card-body">
                        <!-- Bot Status Cards -->
                        <div class="row g-3 mb-4" id="botStatusCards">
                            <div class="col-md-3">
                                <div class="card bg-light">
                                    <div class="card-body text-center">
                                        <i class="bi bi-activity text-primary fs-1 mb-2"></i>
                                        <h6 class="card-title">Status</h6>
                                        <span class="badge bg-secondary" id="botStatus">Loading...</span>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="card bg-light">
                                    <div class="card-body text-center">
                                        <i class="bi bi-clock text-info fs-1 mb-2"></i>
                                        <h6 class="card-title">Last Run</h6>
                                        <small class="text-muted" id="lastRun">Loading...</small>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="card bg-light">
                                    <div class="card-body text-center">
                                        <i class="bi bi-exclamation-triangle text-warning fs-1 mb-2"></i>
                                        <h6 class="card-title">Recent Errors</h6>
                                        <span class="badge bg-warning" id="errorCount">0</span>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="card bg-light">
                                    <div class="card-body text-center">
                                        <i class="bi bi-file-text text-success fs-1 mb-2"></i>
                                        <h6 class="card-title">Log Size</h6>
                                        <small class="text-muted" id="logSize">0 KB</small>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Bot Metrics -->
                        <div class="row g-3 mb-4" id="botMetrics">
                            <div class="col-md-2">
                                <div class="text-center">
                                    <h4 class="text-primary mb-0" id="totalSyncs">0</h4>
                                    <small class="text-muted">Total Syncs</small>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="text-center">
                                    <h4 class="text-success mb-0" id="successfulSyncs">0</h4>
                                    <small class="text-muted">Successful</small>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="text-center">
                                    <h4 class="text-danger mb-0" id="failedSyncs">0</h4>
                                    <small class="text-muted">Failed</small>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="text-center">
                                    <h4 class="text-info mb-0" id="usersUploaded">0</h4>
                                    <small class="text-muted">Users Uploaded</small>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="text-center">
                                    <h4 class="text-warning mb-0" id="enrollments">0</h4>
                                    <small class="text-muted">Enrollments</small>
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="text-center">
                                    <h4 class="text-secondary mb-0" id="avgSyncTime">0s</h4>
                                    <small class="text-muted">Avg Time</small>
                                </div>
                            </div>
                        </div>

                        <!-- Recent Logs -->
                        <div class="row">
                            <div class="col-12">
                                <h6 class="mb-3">
                                    <i class="bi bi-list-ul me-2"></i>Recent Logs
                                    <button type="button" class="btn btn-sm btn-outline-secondary ms-2" onclick="refreshLogs()">
                                        <i class="bi bi-arrow-clockwise me-1"></i>Refresh Logs
                                    </button>
                                </h6>
                                <div class="card bg-dark text-light">
                                    <div class="card-body" style="max-height: 300px; overflow-y: auto; font-family: 'Courier New', monospace; font-size: 0.85rem;">
                                        <div id="botLogs">
                                            <div class="text-center text-muted">
                                                <i class="bi bi-hourglass-split fs-3 d-block mb-2"></i>
                                                Loading logs...
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Cleanup Section -->
        <div class="row g-4 mb-4">
            <div class="col-12">
                <div class="card card-modern">
                    <div class="card-header">
                        <h5 class="mb-0" style="color: #2c3e50; font-weight: 600;">
                            <i class="bi bi-trash3 me-2" style="color: var(--brand-primary);"></i>Cleanup Unverified Users
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php
                        // Handle cleanup action
                        if (isset($_POST['action']) && $_POST['action'] === 'cleanup_unverified') {
                            $result = cleanupUnverifiedUsers();
                            if ($result['success']) {
                                echo '<div class="alert alert-success">' . $result['message'] . '</div>';
                                // Refresh stats
                                $stats = getDashboardStats();
                            } else {
                                echo '<div class="alert alert-danger">' . $result['message'] . '</div>';
                            }
                        }
                        
                        // Get unverified user stats
                        $unverifiedStats = getUnverifiedUserStats();
                        ?>
                        
                        <div class="row">
                            <div class="col-md-8">
                                <p class="text-muted mb-3">
                                    Users who don't verify their email within 24 hours are automatically deleted to allow re-registration.
                                </p>
                                
                                <div class="row g-3">
                                    <div class="col-sm-6">
                                        <div class="d-flex align-items-center">
                                            <i class="bi bi-clock-history text-warning me-2"></i>
                                            <div>
                                                <small class="text-muted">Total Unverified</small>
                                                <div class="fw-bold"><?php echo $unverifiedStats['total_unverified']; ?></div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-sm-6">
                                        <div class="d-flex align-items-center">
                                            <i class="bi bi-exclamation-triangle text-danger me-2"></i>
                                            <div>
                                                <small class="text-muted">Expired (>24h)</small>
                                                <div class="fw-bold"><?php echo $unverifiedStats['expired_unverified']; ?></div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-4 text-end">
                                <?php if ($unverifiedStats['expired_unverified'] > 0): ?>
                                    <form method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to delete <?php echo $unverifiedStats['expired_unverified']; ?> unverified users? This action cannot be undone.');">
                                        <input type="hidden" name="action" value="cleanup_unverified">
                                        <button type="submit" class="btn btn-danger">
                                            <i class="bi bi-trash3 me-1"></i>Clean Up Now
                                        </button>
                                    </form>
                                <?php else: ?>
                                    <button type="button" class="btn btn-outline-secondary" disabled>
                                        <i class="bi bi-check-circle me-1"></i>No Cleanup Needed
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <?php if ($unverifiedStats['total_unverified'] > 0): ?>
                        <div class="mt-3">
                            <small class="text-muted">Age breakdown:</small>
                            <div class="d-flex gap-3 mt-1">
                                <span class="badge bg-primary">Last hour: <?php echo $unverifiedStats['age_breakdown']['last_hour']; ?></span>
                                <span class="badge bg-info">Last 6h: <?php echo $unverifiedStats['age_breakdown']['last_6_hours']; ?></span>
                                <span class="badge bg-warning">Last 12h: <?php echo $unverifiedStats['age_breakdown']['last_12_hours']; ?></span>
                                <span class="badge bg-danger">Last 24h: <?php echo $unverifiedStats['age_breakdown']['last_24_hours']; ?></span>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

		<!-- All Students Table with filters and modal trigger -->
        <div class="row g-4">
			<div class="col-12">
				<div class="card card-modern">
					<div class="card-header">
						<div class="d-flex flex-wrap justify-content-between align-items-center gap-2">
							<h5 class="card-title mb-0" style="color: #2c3e50; font-weight: 600;">
								<i class="bi bi-people me-2" style="color: var(--brand-primary);"></i>All Students
								<span class="badge bg-secondary ms-2"><?= $stats['total_students'] ?></span>
							</h5>
							<div class="d-flex gap-2">
								<button class="btn btn-sm btn-outline-primary" type="button" data-bs-toggle="collapse" data-bs-target="#filterCollapse">
									<i class="bi bi-funnel me-1"></i>Filters
									<?php if ($filterGrade || $filterSubject || $filterEmailStatus || $filterDateFrom || $filterDateTo || $filterUsername || $filterId || $q): ?>
										<span class="badge bg-danger ms-1">Active</span>
									<?php endif; ?>
								</button>
								<button class="btn btn-sm btn-gradient text-white" data-bs-toggle="modal" data-bs-target="#createStudentModal">
									<i class="bi bi-plus-lg me-1"></i>Add Student
								</button>
							</div>
						</div>
						
						<!-- Collapsible Advanced Filters Section -->
						<div class="collapse mt-3 <?php echo ($filterGrade || $filterSubject || $filterEmailStatus || $filterDateFrom || $filterDateTo || $filterUsername || $filterId || $q) ? 'show' : ''; ?>" id="filterCollapse">
							<div class="p-3" style="background: #f8f9fa; border-radius: 8px; border: 1px solid #dee2e6;">
								<form method="get" action="dashboard.php" class="row g-3">
									<!-- Row 1: Main Filters -->
									<div class="col-md-2">
										<label class="form-label small mb-1"><i class="bi bi-mortarboard me-1"></i>Grade</label>
										<select class="form-select form-select-sm" name="grade" id="gradeFilter">
											<option value="0">All Grades</option>
											<?php foreach ($grades as $g): ?>
												<option value="<?= (int)$g['id']; ?>" <?= $filterGrade===(int)$g['id']?'selected':''; ?>><?= htmlspecialchars($g['name']); ?></option>
											<?php endforeach; ?>
										</select>
									</div>
									
									<div class="col-md-2">
										<label class="form-label small mb-1"><i class="bi bi-book me-1"></i>Subject</label>
										<select class="form-select form-select-sm" name="subject" id="subjectFilter">
											<option value="0">All Subjects</option>
											<?php 
											$subjects = $filterGrade > 0 ? getSubjectsForGrade($filterGrade) : getAllSubjects();
											foreach ($subjects as $subject): ?>
												<option value="<?= (int)$subject['id']; ?>" <?= $filterSubject===(int)$subject['id']?'selected':''; ?>><?= htmlspecialchars($subject['name']); ?></option>
											<?php endforeach; ?>
										</select>
									</div>
									
									<div class="col-md-2">
										<label class="form-label small mb-1"><i class="bi bi-envelope me-1"></i>Email Status</label>
										<select class="form-select form-select-sm" name="email_status">
											<option value="">All</option>
											<option value="verified" <?= $filterEmailStatus==='verified'?'selected':''; ?>>Verified</option>
											<option value="pending" <?= $filterEmailStatus==='pending'?'selected':''; ?>>Pending</option>
										</select>
									</div>
									
									<div class="col-md-3">
										<label class="form-label small mb-1"><i class="bi bi-search me-1"></i>Search</label>
										<input type="text" class="form-control form-control-sm" name="q" placeholder="Name or email..." value="<?= htmlspecialchars($q); ?>" />
									</div>
									
									<div class="col-md-3">
										<label class="form-label small mb-1">&nbsp;</label>
										<div class="d-flex gap-2">
											<button type="submit" class="btn btn-sm btn-primary w-100">
												<i class="bi bi-search me-1"></i>Apply
											</button>
											<a href="dashboard.php" class="btn btn-sm btn-outline-secondary" title="Clear Filters">
												<i class="bi bi-x-lg"></i>
											</a>
										</div>
									</div>
									
									<!-- Row 2: Advanced Filters (Collapsible) -->
									<div class="col-12">
										<div class="accordion accordion-flush" id="advancedFiltersAccordion">
											<div class="accordion-item border-0">
												<h2 class="accordion-header">
													<button class="accordion-button collapsed p-2 bg-transparent" type="button" data-bs-toggle="collapse" data-bs-target="#advancedFilters">
														<small><i class="bi bi-sliders me-1"></i>More Filters</small>
													</button>
												</h2>
												<div id="advancedFilters" class="accordion-collapse collapse">
													<div class="accordion-body px-0 pt-3">
														<div class="row g-3">
															<div class="col-md-2">
																<label class="form-label small mb-1"><i class="bi bi-calendar me-1"></i>From Date</label>
																<input type="date" class="form-control form-control-sm" name="date_from" value="<?= htmlspecialchars($filterDateFrom); ?>" />
															</div>
															
															<div class="col-md-2">
																<label class="form-label small mb-1"><i class="bi bi-calendar-check me-1"></i>To Date</label>
																<input type="date" class="form-control form-control-sm" name="date_to" value="<?= htmlspecialchars($filterDateTo); ?>" />
															</div>
															
															<div class="col-md-2">
																<label class="form-label small mb-1"><i class="bi bi-person-badge me-1"></i>Username</label>
																<input type="text" class="form-control form-control-sm" name="username" placeholder="Username" value="<?= htmlspecialchars($filterUsername); ?>" />
															</div>
															
															<div class="col-md-2">
																<label class="form-label small mb-1"><i class="bi bi-hash me-1"></i>Student ID</label>
																<input type="number" class="form-control form-control-sm" name="id" placeholder="ID" value="<?= htmlspecialchars($filterId); ?>" />
															</div>
														</div>
													</div>
												</div>
											</div>
										</div>
									</div>
								</form>
							</div>
						</div>
                    </div>
                    <div class="card-body">
						<?php if (empty($students)): ?>
							<div class="text-center py-4">
								<i class="bi bi-search text-muted" style="font-size: 3rem;"></i>
								<p class="text-muted mt-2">No students found.</p>
							</div>
						<?php else: ?>
							<div class="d-flex justify-content-between align-items-center mb-3">
								<span class="text-muted" id="results-count">
									<i class="bi bi-info-circle me-1"></i>
									Showing <?php echo count($students); ?> student<?php echo count($students) !== 1 ? 's' : ''; ?>
								</span>
								<div class="btn-group" role="group">
									<button type="button" class="btn btn-outline-primary btn-sm dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">
										<i class="bi bi-download me-1"></i>Export
									</button>
									<ul class="dropdown-menu dropdown-menu-end">
										<li>
											<h6 class="dropdown-header"><i class="bi bi-download me-1"></i>Export Students</h6>
										</li>
										<li><a class="dropdown-item" href="export_students.php?format=csv&<?= http_build_query($_GET) ?>">
											<i class="bi bi-filetype-csv text-success me-2"></i>CSV (Excel Compatible)
										</a></li>
										<li><hr class="dropdown-divider"></li>
										<li><a class="dropdown-item" href="export_students.php?format=html&<?= http_build_query($_GET) ?>" target="_blank">
											<i class="bi bi-file-earmark-pdf text-danger me-2"></i>PDF (Print to Save)
										</a></li>
										<li><a class="dropdown-item" href="export_students.php?format=html&<?= http_build_query($_GET) ?>" target="_blank">
											<i class="bi bi-file-earmark-code text-primary me-2"></i>HTML (with Letterhead)
										</a></li>
									</ul>
								</div>
							</div>
							<div class="table-responsive">
								<table class="table table-striped align-middle">
									<thead>
										<tr>
											<th>ID</th>
											<th>Name</th>
											<th>Email</th>
											<th>Grade</th>
											<th>Username</th>
											<th>Subjects</th>
											<th>Email Status</th>
											<th>Payment Status</th>
											<th>Moodle Status</th>
											<th>Created</th>
											<th class="text-end">Actions</th>
										</tr>
									</thead>
									<tbody>
										<?php foreach ($students as $s): ?>
										<?php 
											// Get subjects for this student
											$subjects = getSubjectsForStudent($s['id']);
											$subjectNames = array_map(function($subject) { return $subject['name']; }, $subjects);
											$subjectsString = implode(', ', $subjectNames);
										?>
										<tr>
											<td><?php echo (int)$s['id']; ?></td>
											<td><?php echo htmlspecialchars($s['full_name'] ?? ($s['first_name'] ?? '') . ' ' . ($s['last_name'] ?? '')); ?></td>
											<td><?php echo htmlspecialchars($s['email']); ?></td>
											<td><?php echo htmlspecialchars($s['grade_name']); ?></td>
											<td><span class="badge bg-secondary"><?php echo htmlspecialchars(formatUsernameForMoodle($s['username'])); ?></span></td>
											<td>
												<?php if (!empty($subjects)): ?>
													<div class="subjects-container">
														<?php foreach ($subjects as $subject): ?>
															<span class="badge bg-primary" title="<?php echo htmlspecialchars($subject['name']); ?>">
																<?php echo htmlspecialchars($subject['name']); ?>
															</span>
														<?php endforeach; ?>
													</div>
												<?php else: ?>
													<span class="text-muted small">No subjects</span>
												<?php endif; ?>
											</td>
											<td>
												<div class="d-flex align-items-center gap-2">
													<?php if (isset($s['email_verified']) && $s['email_verified']): ?>
														<span class="badge bg-success">
															<i class="bi bi-check-circle me-1"></i>Verified
														</span>
													<?php else: ?>
														<span class="badge bg-warning">
															<i class="bi bi-exclamation-triangle me-1"></i>Pending
														</span>
													<?php endif; ?>
													<button class="btn btn-sm btn-outline-secondary toggle-email-status" 
															data-student-id="<?php echo (int)$s['id']; ?>" 
															data-current-status="<?php echo $s['email_verified'] ? '1' : '0'; ?>"
															title="Toggle Email Status">
														<i class="bi bi-arrow-repeat"></i>
													</button>
												</div>
											</td>
											<td>
												<div class="d-flex align-items-center gap-2">
													<?php 
													// Check payment status for this student
													$paymentStatus = getStudentPaymentStatus($s['id']);
													if ($paymentStatus['has_paid']): ?>
														<span class="badge bg-success">
															<i class="bi bi-credit-card me-1"></i>Paid
														</span>
													<?php elseif ($paymentStatus['has_pending']): ?>
														<span class="badge bg-warning">
															<i class="bi bi-clock me-1"></i>Pending
														</span>
													<?php else: ?>
														<span class="badge bg-danger">
															<i class="bi bi-x-circle me-1"></i>Unpaid
														</span>
													<?php endif; ?>
													<button class="btn btn-sm btn-outline-secondary toggle-payment-status" 
															data-student-id="<?php echo (int)$s['id']; ?>" 
															data-current-status="<?php echo $paymentStatus['has_paid'] ? 'paid' : ($paymentStatus['has_pending'] ? 'pending' : 'unpaid'); ?>"
															title="Toggle Payment Status">
														<i class="bi bi-arrow-repeat"></i>
													</button>
												</div>
											</td>
											<td>
												<?php 
												$moodleStatus = $s['moodle_status'] ?? 'active';
												if ($moodleStatus === 'active'): ?>
													<span class="badge bg-success">
														<i class="bi bi-check-circle me-1"></i>Active
													</span>
												<?php elseif ($moodleStatus === 'suspended'): ?>
													<span class="badge bg-danger">
														<i class="bi bi-x-circle me-1"></i>Suspended
													</span>
												<?php else: ?>
													<span class="badge bg-secondary">
														<i class="bi bi-question-circle me-1"></i>Unknown
													</span>
												<?php endif; ?>
											</td>
											<td><?php echo htmlspecialchars($s['created_at']); ?></td>
											<td class="text-end">
												<div class="btn-group" role="group">
													<a class="btn btn-sm btn-outline-primary action-btn" href="edit.php?id=<?php echo (int)$s['id']; ?>" title="Edit Student">
														<i class="bi bi-pencil"></i>
													</a>
													<a class="btn btn-sm btn-outline-danger action-btn" href="delete.php?id=<?php echo (int)$s['id']; ?>" title="Delete Student" onclick="return confirm('Are you sure you want to delete this student? This action cannot be undone.');">
														<i class="bi bi-trash"></i>
													</a>
												</div>
											</td>
										</tr>
										<?php endforeach; ?>
									</tbody>
								</table>
							</div>
							
							<!-- Pagination Controls -->
							<?php if ($totalPages > 1): ?>
							<div class="d-flex justify-content-between align-items-center mt-4">
								<div class="pagination-info">
									<small class="text-muted">
										Showing <?php echo $offset + 1; ?> to <?php echo min($offset + $studentsPerPage, $totalStudents); ?> of <?php echo $totalStudents; ?> students
									</small>
								</div>
								<nav aria-label="Students pagination">
									<ul class="pagination pagination-sm mb-0">
										<!-- Previous Page -->
										<?php if ($currentPage > 1): ?>
											<li class="page-item">
												<a class="page-link" href="?page=<?php echo $currentPage - 1; ?><?php echo !empty($_GET) ? '&' . http_build_query(array_filter($_GET, function($k) { return $k !== 'page'; }, ARRAY_FILTER_USE_KEY)) : ''; ?>">
													<i class="bi bi-chevron-left"></i>
												</a>
											</li>
										<?php else: ?>
											<li class="page-item disabled">
												<span class="page-link"><i class="bi bi-chevron-left"></i></span>
											</li>
										<?php endif; ?>
										
										<!-- Page Numbers -->
										<?php
										$startPage = max(1, $currentPage - 2);
										$endPage = min($totalPages, $currentPage + 2);
										
										// Show first page if not in range
										if ($startPage > 1): ?>
											<li class="page-item">
												<a class="page-link" href="?page=1<?php echo !empty($_GET) ? '&' . http_build_query(array_filter($_GET, function($k) { return $k !== 'page'; }, ARRAY_FILTER_USE_KEY)) : ''; ?>">1</a>
											</li>
											<?php if ($startPage > 2): ?>
												<li class="page-item disabled"><span class="page-link">...</span></li>
											<?php endif; ?>
										<?php endif; ?>
										
										<?php for ($i = $startPage; $i <= $endPage; $i++): ?>
											<li class="page-item <?php echo $i == $currentPage ? 'active' : ''; ?>">
												<a class="page-link" href="?page=<?php echo $i; ?><?php echo !empty($_GET) ? '&' . http_build_query(array_filter($_GET, function($k) { return $k !== 'page'; }, ARRAY_FILTER_USE_KEY)) : ''; ?>">
													<?php echo $i; ?>
												</a>
											</li>
										<?php endfor; ?>
										
										<!-- Show last page if not in range -->
										<?php if ($endPage < $totalPages): ?>
											<?php if ($endPage < $totalPages - 1): ?>
												<li class="page-item disabled"><span class="page-link">...</span></li>
											<?php endif; ?>
											<li class="page-item">
												<a class="page-link" href="?page=<?php echo $totalPages; ?><?php echo !empty($_GET) ? '&' . http_build_query(array_filter($_GET, function($k) { return $k !== 'page'; }, ARRAY_FILTER_USE_KEY)) : ''; ?>"><?php echo $totalPages; ?></a>
											</li>
										<?php endif; ?>
										
										<!-- Next Page -->
										<?php if ($currentPage < $totalPages): ?>
											<li class="page-item">
												<a class="page-link" href="?page=<?php echo $currentPage + 1; ?><?php echo !empty($_GET) ? '&' . http_build_query(array_filter($_GET, function($k) { return $k !== 'page'; }, ARRAY_FILTER_USE_KEY)) : ''; ?>">
													<i class="bi bi-chevron-right"></i>
												</a>
											</li>
										<?php else: ?>
											<li class="page-item disabled">
												<span class="page-link"><i class="bi bi-chevron-right"></i></span>
											</li>
										<?php endif; ?>
									</ul>
								</nav>
							</div>
							<?php endif; ?>
						<?php endif; ?>
					</div>
				</div>
			</div>
		</div>

		<!-- Charts row moved below -->
		<div class="row g-4 mt-1">
			<!-- Students by Grade Chart -->
			<div class="col-md-8">
				<div class="card card-modern mb-4" style="height: 400px;">
					<div class="card-header">
						<h5 class="card-title mb-0" style="color: #2c3e50; font-weight: 600;">
							<i class="bi bi-bar-chart me-2" style="color: var(--brand-primary);"></i>Students by Grade
						</h5>
					</div>
					<div class="card-body" style="height: 320px;">
						<canvas id="studentsChart" height="100"></canvas>
					</div>
				</div>
				<div class="card card-modern" style="height: 400px;">
					<div class="card-header">
						<h5 class="card-title mb-0" style="color: #2c3e50; font-weight: 600;">
							<i class="bi bi-pie-chart me-2" style="color: var(--brand-primary);"></i>Grade Distribution
						</h5>
					</div>
					<div class="card-body" style="height: 320px;">
						<canvas id="gradePieChart" height="100"></canvas>
                    </div>
                </div>
            </div>
            
            <!-- Recent Students -->
            <div class="col-md-4">
				<div class="card card-modern" style="height: 840px;">
					<div class="card-header">
						<h5 class="card-title mb-0" style="color: #2c3e50; font-weight: 600;">
							<i class="bi bi-clock-history me-2" style="color: var(--brand-primary);"></i>Recent Registrations
						</h5>
                    </div>
                    <div class="card-body">
						<?php $recent = array_slice($students, 0, 10); ?>
						<?php if (empty($recent)): ?>
                            <p class="text-muted">No students registered yet.</p>
                        <?php else: ?>
                            <div class="list-group list-group-flush">
								<?php foreach ($recent as $student): ?>
                                    <div class="list-group-item px-0">
                                        <div class="d-flex w-100 justify-content-between">
                                            <h6 class="mb-1"><?php echo htmlspecialchars($student['full_name'] ?? ($student['first_name'] ?? '') . ' ' . ($student['last_name'] ?? '')); ?></h6>
                                            <small><?php echo date('M j', strtotime($student['created_at'])); ?></small>
                                        </div>
                                        <p class="mb-1 text-muted"><?php echo htmlspecialchars($student['grade_name']); ?></p>
                                        <small><?php echo htmlspecialchars($student['email']); ?></small>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

		<!-- Growth Chart -->
		<div class="row g-4 mt-1">
			<div class="col-12">
				<div class="card card-modern" style="height: 400px;">
					<div class="card-header">
						<h5 class="card-title mb-0" style="color: #2c3e50; font-weight: 600;">
							<i class="bi bi-activity me-2" style="color: var(--brand-primary);"></i>Registrations - Last 12 Months
						</h5>
					</div>
					<div class="card-body" style="height: 320px;">
						<canvas id="growthChart" height="80"></canvas>
					</div>
				</div>
			</div>
		</div>
	</div>

	<!-- Create Student Modal -->
	<div class="modal fade" id="createStudentModal" tabindex="-1" aria-hidden="true">
		<div class="modal-dialog modal-lg modal-dialog-scrollable">
			<div class="modal-content">
				<div class="modal-header gradient">
					<h5 class="modal-title"><i class="bi bi-person-plus me-1"></i>Create Student</h5>
					<button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
				</div>
				<div class="modal-body">
					<div id="modalAlert"></div>
					
					<!-- Form Header -->
					<div class="text-center mb-4">
						<h6 class="text-primary mb-2">
							<i class="bi bi-person-plus-fill me-2"></i>
							Add New Student
						</h6>
						<p class="text-muted mb-0 small">Create a student account with all required information</p>
					</div>

					<form id="createStudentForm">
						<!-- Personal Information Section -->
						<div class="form-section mb-3">
							<h6 class="section-title text-primary mb-3">
								<i class="bi bi-person-fill me-2"></i>Personal Information
							</h6>
							<div class="row g-3">
								<div class="col-md-6">
									<label class="form-label fw-semibold">First Name *</label>
									<div class="input-icon">
										<i class="bi bi-person"></i>
										<input class="form-control form-control-lg" name="first_name" placeholder="e.g. Jane" required />
									</div>
								</div>
								<div class="col-md-6">
									<label class="form-label fw-semibold">Last Name *</label>
									<div class="input-icon">
										<i class="bi bi-person"></i>
										<input class="form-control form-control-lg" name="last_name" placeholder="e.g. Doe" required />
									</div>
								</div>
							</div>
						</div>

						<!-- Contact Information Section -->
						<div class="form-section mb-3">
							<h6 class="section-title text-primary mb-3">
								<i class="bi bi-telephone-fill me-2"></i>Contact Information
							</h6>
							<div class="row g-3">
								<div class="col-md-6">
									<label class="form-label fw-semibold">Email Address *</label>
									<div class="input-icon">
										<i class="bi bi-envelope"></i>
										<input type="email" class="form-control form-control-lg" name="email" placeholder="you@example.com" required />
									</div>
									<div class="form-text">
										<i class="bi bi-info-circle me-1"></i>
										Student's email for login and notifications
									</div>
								</div>
								<div class="col-md-6">
									<label class="form-label fw-semibold">Phone Number *</label>
									<div class="input-icon">
										<i class="bi bi-phone"></i>
										<input type="tel" class="form-control form-control-lg" name="phone" placeholder="e.g. 26876873207" required />
									</div>
									<div class="alert alert-warning py-2 px-3 mt-2 mb-0">
										<i class="bi bi-exclamation-triangle-fill me-1"></i>
										<small><strong>Required:</strong> Must have active MTN MoMo account for payments</small>
									</div>
								</div>
							</div>
						</div>

						<!-- Academic Information Section -->
						<div class="form-section mb-3">
							<h6 class="section-title text-primary mb-3">
								<i class="bi bi-mortarboard-fill me-2"></i>Academic Information
							</h6>
							<div class="row g-3">
								<div class="col-md-6">
									<label class="form-label fw-semibold">Grade Level *</label>
									<div class="input-icon">
										<i class="bi bi-mortarboard"></i>
										<select class="form-select form-select-lg" name="grade_id" id="modalGradeSelect" required>
											<option value="">Select grade...</option>
											<?php foreach ($grades as $g): ?>
												<option value="<?php echo (int)$g['id']; ?>"><?php echo htmlspecialchars($g['name']); ?></option>
											<?php endforeach; ?>
										</select>
									</div>
								</div>
								<div class="col-12">
									<label class="form-label fw-semibold">Select Subjects</label>
									<div id="modalSubjectsContainer" class="border rounded-3 p-3 bg-light" style="min-height: 120px;">
										<div class="text-muted text-center py-4">
											<i class="bi bi-mortarboard fs-1 d-block mb-3 text-primary"></i>
											<h6 class="mb-2">Choose Subjects</h6>
											<p class="mb-0">Please select a grade first to see available subjects</p>
										</div>
									</div>
									<div class="form-text mt-2">
										<i class="bi bi-info-circle me-1"></i>
										Click on subjects to select them. Multiple selections allowed.
									</div>
								</div>
							</div>
						</div>

						<!-- Security Section -->
						<div class="form-section mb-3">
							<h6 class="section-title text-primary mb-3">
								<i class="bi bi-shield-fill me-2"></i>Account Security
							</h6>
							<div class="row g-3">
								<div class="col-md-6">
									<label class="form-label fw-semibold">Password *</label>
									<div class="input-icon">
										<i class="bi bi-shield-lock"></i>
										<input type="password" class="form-control form-control-lg" name="password" minlength="8" placeholder="At least 8 characters" required />
									</div>
									<div class="form-text">
										<i class="bi bi-lightbulb me-1"></i>
										Use a strong password the student can remember
									</div>
								</div>
								<div class="col-md-6">
									<label class="form-label fw-semibold">Confirm Password *</label>
									<div class="input-icon">
										<i class="bi bi-shield-check"></i>
										<input type="password" class="form-control form-control-lg" name="password_confirm" minlength="8" placeholder="Re-type password" required />
									</div>
								</div>
							</div>
						</div>
					</form>
				</div>
				<div class="modal-footer">
					<button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Close</button>
					<button type="button" id="createStudentSubmit" class="btn btn-gradient text-white">Create Student</button>
				</div>
			</div>
		</div>
    </div>
    <?php include __DIR__.'/partials/footer.php'; ?>

    <script>
        // Students by Grade Chart
            const gradeData = <?php echo json_encode($stats['students_by_grade']); ?>;
		const studentsCtx = document.getElementById('studentsChart');
		if (studentsCtx) {
            const labels = gradeData.map(item => item.grade);
            const data = gradeData.map(item => parseInt(item.count));
			new Chart(studentsCtx, { type: 'bar', data: { labels, datasets: [{ label: 'Students', data, backgroundColor: 'rgba(111, 66, 193, 0.4)', borderColor: 'rgba(111, 66, 193, 1)', borderWidth: 1 }] }, options: { responsive: true, maintainAspectRatio: false, scales: { y: { beginAtZero: true, ticks: { stepSize: 1 } } } } });
		}
		// Grade Distribution Pie
		const gradePieCtx = document.getElementById('gradePieChart');
		if (gradePieCtx) {
			const pieLabels = gradeData.map(item => item.grade);
			const pieData = gradeData.map(item => parseInt(item.count));
			const colors = ['#6f42c1','#20c997','#fd7e14','#0dcaf0','#198754','#dc3545','#ffc107','#0d6efd','#6610f2','#6c757d'];
			new Chart(gradePieCtx, { type: 'doughnut', data: { labels: pieLabels, datasets: [{ data: pieData, backgroundColor: colors.slice(0, pieLabels.length) }] }, options: { responsive: true, maintainAspectRatio: false, plugins: { legend: { position: 'bottom' } }, cutout: '55%' } });
		}
		// Growth Chart
		const growthCtx = document.getElementById('growthChart');
		if (growthCtx) { fetch('growth_data.php', { headers: { 'Accept': 'application/json' } }).then(r => r.ok ? r.json() : Promise.reject()).then(({ labels, data }) => { new Chart(growthCtx, { type: 'line', data: { labels, datasets: [{ label: 'Registrations', data, borderColor: 'rgba(32, 201, 151, 1)', backgroundColor: 'rgba(32, 201, 151, 0.2)', fill: true, tension: 0.25 }] }, options: { responsive: true, maintainAspectRatio: false } }); }).catch(() => {}); }

		// Modal: load subjects when grade changes
		const modalGradeSelect = document.getElementById('modalGradeSelect');
		const modalSubjectsContainer = document.getElementById('modalSubjectsContainer');
		
		function renderModalSubjects(items){
			modalSubjectsContainer.innerHTML = '';
			if(!items || !items.length){
				modalSubjectsContainer.innerHTML = `
					<div class="text-muted text-center py-4">
						<i class="bi bi-exclamation-circle fs-1 d-block mb-2"></i>
						No subjects available for this grade yet.
					</div>
				`;
				return;
			}
			
			// Create header with Select All / Clear All buttons
			const buttonRow = document.createElement('div');
			buttonRow.className = 'row mb-3';
			buttonRow.innerHTML = `
				<div class="col-12">
					<div class="d-flex gap-2">
						<button type="button" class="btn btn-sm btn-outline-primary" id="modalSelectAllSubjects">
							<i class="bi bi-check-square me-1"></i>Select All
						</button>
						<button type="button" class="btn btn-sm btn-outline-secondary" id="modalClearAllSubjects">
							<i class="bi bi-square me-1"></i>Clear All
						</button>
					</div>
				</div>
			`;
			modalSubjectsContainer.appendChild(buttonRow);
			
			// Create checkbox grid
			const row = document.createElement('div');
			row.className = 'row g-2';
			
			items.forEach(s => {
				const col = document.createElement('div');
				col.className = 'col-md-6 col-lg-4';
				
				const checkboxDiv = document.createElement('div');
				checkboxDiv.className = 'form-check subject-checkbox p-2';
				
				const checkbox = document.createElement('input');
				checkbox.type = 'checkbox';
				checkbox.className = 'form-check-input';
				checkbox.name = 'subject_ids[]';
				checkbox.value = s.id;
				checkbox.id = 'modal_subject_' + s.id;
				
				const label = document.createElement('label');
				label.className = 'form-check-label w-100';
				label.htmlFor = 'modal_subject_' + s.id;
				label.textContent = s.name + (s.code ? ' ('+s.code+')' : '');
				
				checkboxDiv.appendChild(checkbox);
				checkboxDiv.appendChild(label);
				col.appendChild(checkboxDiv);
				row.appendChild(col);
			});
			
			modalSubjectsContainer.appendChild(row);
			
			// Add event listeners for Select All / Clear All buttons
			document.getElementById('modalSelectAllSubjects').addEventListener('click', function() {
				const checkboxes = modalSubjectsContainer.querySelectorAll('input[name="subject_ids[]"]');
				checkboxes.forEach(cb => cb.checked = true);
			});
			
			document.getElementById('modalClearAllSubjects').addEventListener('click', function() {
				const checkboxes = modalSubjectsContainer.querySelectorAll('input[name="subject_ids[]"]');
				checkboxes.forEach(cb => cb.checked = false);
			});
		}
		
		async function loadModalSubjects(){
			const gid = modalGradeSelect.value;
			modalSubjectsContainer.innerHTML = '';
			if(!gid) return;
			
			// Show loading message
			modalSubjectsContainer.innerHTML = `
				<div class="text-muted text-center py-4">
					<i class="bi bi-hourglass-split fs-1 d-block mb-2"></i>
					Loading subjects...
				</div>
			`;
			
			try{
				const res = await fetch('../register.php?action=subjects&grade_id='+encodeURIComponent(gid), {headers:{'Accept':'application/json'}});
				if(!res.ok) {
					throw new Error('Server error: ' + res.status);
				}
				const data = await res.json();
				renderModalSubjects(data || []);
			}catch(e){
				console.error('Error loading subjects:', e);
				// Fallback: show sample subjects when database is not available
				const sampleSubjects = [
					{id: 1, name: 'Mathematics', code: 'MATH'},
					{id: 2, name: 'English', code: 'ENG'},
					{id: 3, name: 'Science', code: 'SCI'},
					{id: 4, name: 'History', code: 'HIST'},
					{id: 5, name: 'Geography', code: 'GEO'},
					{id: 6, name: 'Art', code: 'ART'},
					{id: 7, name: 'Physical Education', code: 'PE'},
					{id: 8, name: 'Computer Studies', code: 'CS'}
				];
				
				modalSubjectsContainer.innerHTML = `
					<div class="alert alert-warning mb-3">
						<i class="bi bi-exclamation-triangle me-1"></i>
						Database connection issue. Showing sample subjects for testing.
					</div>
				`;
				renderModalSubjects(sampleSubjects);
			}
		}
		
		modalGradeSelect?.addEventListener('change', loadModalSubjects);

		// Modal submit via AJAX
		const createBtn = document.getElementById('createStudentSubmit');
		const formEl = document.getElementById('createStudentForm');
		const modalAlert = document.getElementById('modalAlert');
		function showModalAlert(type, msg){ modalAlert.innerHTML = '<div class="alert alert-'+type+' alert-dismissible fade show" role="alert">'+msg+'<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button></div>'; }
		createBtn?.addEventListener('click', async ()=>{
			const fd = new FormData(formEl);
			createBtn.disabled = true; createBtn.textContent = 'Creating...';
			try{
				const res = await fetch('create_api.php', { method:'POST', body: fd });
				const json = await res.json();
				if(!json.ok){
					const msg = json.errors ? json.errors.map(e=>'<li>'+e+'</li>').join('') : (json.message||'Failed');
					showModalAlert('danger', '<strong>Could not create student:</strong><ul class="mb-0">'+msg+'</ul>');
				} else {
					showModalAlert('success', 'Student created successfully. Reloading...');
					setTimeout(()=>{ window.location.reload(); }, 800);
				}
			}catch(e){ showModalAlert('danger','An unexpected error occurred.'); }
			finally{ createBtn.disabled = false; createBtn.textContent = 'Create Student'; }
		});
    </script>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Auto-filter functionality with pagination support
            const form = document.querySelector('form[method="get"]');
            const filterInputs = form.querySelectorAll('select, input[type="date"], input[name="username"], input[name="id"], input[name="q"]');
            const table = document.querySelector('.table tbody');
            const allRows = Array.from(table.querySelectorAll('tr'));
            const resultsCount = document.querySelector('#results-count');
            
            // Check if we have pagination (more than 20 students)
            const totalStudents = <?php echo $totalStudents; ?>;
            const hasPagination = totalStudents > 20;
            
            // Store all student data for filtering (only if no pagination)
            let studentData = [];
            if (!hasPagination) {
                studentData = allRows.map(row => {
                    const cells = row.querySelectorAll('td');
                    const subjects = Array.from(row.querySelectorAll('.subjects-container .badge')).map(badge => badge.textContent.trim());
                    return {
                        element: row,
                        id: cells[0]?.textContent.trim() || '',
                        name: cells[1]?.textContent.trim() || '',
                        email: cells[2]?.textContent.trim() || '',
                        grade: cells[3]?.textContent.trim() || '',
                        username: cells[4]?.textContent.trim() || '',
                        subjects: subjects,
                        emailStatus: cells[6]?.textContent.trim() || '',
                        created: cells[7]?.textContent.trim() || ''
                    };
                });
            }
            
            // Debounce function to prevent too many requests
            function debounce(func, wait) {
                let timeout;
                return function executedFunction(...args) {
                    const later = () => {
                        clearTimeout(timeout);
                        func(...args);
                    };
                    clearTimeout(timeout);
                    timeout = setTimeout(later, wait);
                };
            }
            
            // Filtering function - server-side for pagination, client-side for small datasets
            function autoFilter() {
                if (hasPagination) {
                    // Server-side filtering with page reload
                    form.submit();
                } else {
                    // Client-side filtering
                    filterTable();
                }
            }
            
            // Debounced filtering for text inputs (300ms delay)
            const debouncedFilter = debounce(() => {
                if (hasPagination) {
                    // Server-side filtering with page reload
                    form.submit();
                } else {
                    // Client-side filtering
                    filterTable();
                }
            }, 300);
            
            // Main filtering function (only for non-pagination mode)
            function filterTable() {
                if (hasPagination) {
                    return; // Skip client-side filtering when pagination is active
                }
                
                const filters = {
                    grade: document.querySelector('select[name="grade"]')?.value || '0',
                    subject: document.querySelector('select[name="subject"]')?.value || '0',
                    emailStatus: document.querySelector('select[name="email_status"]')?.value || '',
                    dateFrom: document.querySelector('input[name="date_from"]')?.value || '',
                    dateTo: document.querySelector('input[name="date_to"]')?.value || '',
                    username: document.querySelector('input[name="username"]')?.value.toLowerCase().trim() || '',
                    id: document.querySelector('input[name="id"]')?.value.trim() || '',
                    search: document.querySelector('input[name="q"]')?.value.toLowerCase().trim() || ''
                };
                
                let visibleCount = 0;
                
                studentData.forEach(student => {
                    let showRow = true;
                    
                    // Grade filter
                    if (filters.grade !== '0') {
                        const gradeSelect = document.querySelector('select[name="grade"]');
                        const selectedGradeText = gradeSelect?.options[gradeSelect.selectedIndex]?.text || '';
                        if (student.grade !== selectedGradeText) {
                            showRow = false;
                        }
                    }
                    
                    // Subject filter
                    if (filters.subject !== '0' && showRow) {
                        const subjectSelect = document.querySelector('select[name="subject"]');
                        const selectedSubjectText = subjectSelect?.options[subjectSelect.selectedIndex]?.text || '';
                        if (!student.subjects.includes(selectedSubjectText)) {
                            showRow = false;
                        }
                    }
                    
                    // Email status filter
                    if (filters.emailStatus !== '' && showRow) {
                        const statusText = filters.emailStatus === 'verified' ? 'Verified' : 'Pending';
                        if (!student.emailStatus.includes(statusText)) {
                            showRow = false;
                        }
                    }
                    
                    // Date filters
                    if (filters.dateFrom !== '' && showRow) {
                        const studentDate = new Date(student.created);
                        const fromDate = new Date(filters.dateFrom);
                        if (studentDate < fromDate) {
                            showRow = false;
                        }
                    }
                    
                    if (filters.dateTo !== '' && showRow) {
                        const studentDate = new Date(student.created);
                        const toDate = new Date(filters.dateTo);
                        toDate.setHours(23, 59, 59, 999); // End of day
                        if (studentDate > toDate) {
                            showRow = false;
                        }
                    }
                    
                    // Username filter
                    if (filters.username !== '' && showRow) {
                        if (!student.username.toLowerCase().includes(filters.username)) {
                            showRow = false;
                        }
                    }
                    
                    // ID filter
                    if (filters.id !== '' && showRow) {
                        if (student.id !== filters.id) {
                            showRow = false;
                        }
                    }
                    
                    // General search filter
                    if (filters.search !== '' && showRow) {
                        const searchText = filters.search;
                        if (!student.name.toLowerCase().includes(searchText) && 
                            !student.email.toLowerCase().includes(searchText)) {
                            showRow = false;
                        }
                    }
                    
                    // Show/hide row
                    if (showRow) {
                        student.element.style.display = '';
                        visibleCount++;
                    } else {
                        student.element.style.display = 'none';
                    }
                });
                
                // Update results count
                if (resultsCount) {
                    const studentText = visibleCount === 1 ? 'student' : 'students';
                    resultsCount.innerHTML = `
                        <i class="bi bi-info-circle me-1"></i>
                        Showing ${visibleCount} ${studentText}
                    `;
                }
                
                // Update active filters display
                updateActiveFiltersDisplay(filters);
            }
            
            // Update active filters display
            function updateActiveFiltersDisplay(filters) {
                const activeFilters = [];
                
                if (filters.grade !== '0') {
                    const gradeSelect = document.querySelector('select[name="grade"]');
                    const gradeText = gradeSelect?.options[gradeSelect.selectedIndex]?.text || '';
                    activeFilters.push(`Grade: ${gradeText}`);
                }
                
                if (filters.subject !== '0') {
                    const subjectSelect = document.querySelector('select[name="subject"]');
                    const subjectText = subjectSelect?.options[subjectSelect.selectedIndex]?.text || '';
                    activeFilters.push(`Subject: ${subjectText}`);
                }
                
                if (filters.emailStatus !== '') {
                    activeFilters.push(`Email: ${filters.emailStatus.charAt(0).toUpperCase() + filters.emailStatus.slice(1)}`);
                }
                
                if (filters.dateFrom !== '') {
                    activeFilters.push(`From: ${filters.dateFrom}`);
                }
                
                if (filters.dateTo !== '') {
                    activeFilters.push(`To: ${filters.dateTo}`);
                }
                
                if (filters.username !== '') {
                    activeFilters.push(`Username: ${filters.username}`);
                }
                
                if (filters.id !== '') {
                    activeFilters.push(`ID: ${filters.id}`);
                }
                
                if (filters.search !== '') {
                    activeFilters.push(`Search: ${filters.search}`);
                }
                
                // Update or create active filters display
                let activeFiltersDiv = document.querySelector('.active-filters-display');
                if (!activeFiltersDiv) {
                    activeFiltersDiv = document.createElement('div');
                    activeFiltersDiv.className = 'active-filters-display alert alert-info d-flex align-items-center mb-3';
                    activeFiltersDiv.style.cssText = 'background: linear-gradient(135deg, rgba(111,66,193,0.1), rgba(32,201,151,0.1)); border: 1px solid rgba(111,66,193,0.2); border-radius: 0.75rem; padding: 1rem;';
                    
                    const cardBody = document.querySelector('.card-body');
                    if (cardBody) {
                        cardBody.insertBefore(activeFiltersDiv, cardBody.firstChild);
                    }
                }
                
                if (activeFilters.length > 0) {
                    activeFiltersDiv.innerHTML = `
                        <div class="d-flex align-items-center">
                            <i class="bi bi-funnel-fill me-2" style="color: var(--brand-primary);"></i>
                            <strong style="color: #2c3e50;">Active Filters:</strong>
                            <span class="ms-2" style="color: #495057;">${activeFilters.join(' • ')}</span>
                        </div>
                    `;
                    activeFiltersDiv.style.display = 'flex';
                } else {
                    activeFiltersDiv.style.display = 'none';
                }
            }
            
            // Add event listeners to all filter inputs
            filterInputs.forEach(input => {
                if (input.type === 'text' || input.name === 'q') {
                    // For text inputs, use input event with debouncing
                    input.addEventListener('input', debouncedFilter);
                } else {
                    // For selects and date inputs, use change event
                    input.addEventListener('change', autoFilter);
                }
            });
            
            // Dynamic subject filter based on grade selection
            const gradeFilter = document.getElementById('gradeFilter');
            const subjectFilter = document.getElementById('subjectFilter');
            
            if (gradeFilter && subjectFilter) {
                gradeFilter.addEventListener('change', function() {
                    const gradeId = this.value;
                    
                    // Clear subject filter when grade changes
                    subjectFilter.innerHTML = '<option value="0">All subjects</option>';
                    
                    if (gradeId > 0) {
                        // Fetch subjects for the selected grade
                        fetch(`../register.php?action=subjects&grade_id=${gradeId}`, {
                            headers: { 'Accept': 'application/json' }
                        })
                        .then(response => response.json())
                        .then(subjects => {
                            subjects.forEach(subject => {
                                const option = document.createElement('option');
                                option.value = subject.id;
                                option.textContent = subject.name;
                                subjectFilter.appendChild(option);
                            });
                            // Trigger filtering after subjects are loaded
                            filterTable();
                        })
                        .catch(error => {
                            console.error('Error loading subjects:', error);
                        });
                    } else {
                        // Load all subjects when no grade is selected
                        fetch(`../register.php?action=all_subjects`, {
                            headers: { 'Accept': 'application/json' }
                        })
                        .then(response => response.json())
                        .then(subjects => {
                            subjects.forEach(subject => {
                                const option = document.createElement('option');
                                option.value = subject.id;
                                option.textContent = subject.name;
                                subjectFilter.appendChild(option);
                            });
                            // Trigger filtering after subjects are loaded
                            filterTable();
                        })
                        .catch(error => {
                            console.error('Error loading all subjects:', error);
                        });
                    }
                });
            }
            
            // Initialize filtering on page load
            filterTable();
        });
        
        // Export functions
        function exportTable(format) {
            // Get all filtered data from server instead of just visible rows
            getAllFilteredStudents().then(allStudents => {
                const timestamp = new Date().toISOString().slice(0, 19).replace(/:/g, '-');
                const filename = `students_export_${timestamp}`;
                
                switch(format) {
                    case 'csv':
                        exportToCSV(allStudents, filename);
                        break;
                    case 'excel':
                        exportToExcel(allStudents, filename);
                        break;
                    case 'pdf':
                        exportToPDF(allStudents, filename);
                        break;
                    case 'html':
                        exportToHTML(allStudents, filename);
                        break;
                }
            }).catch(error => {
                console.error('Error getting filtered students:', error);
                alert('Error loading data for export. Please try again.');
            });
        }
        
        function getVisibleStudents() {
            const visibleRows = Array.from(document.querySelectorAll('.table tbody tr')).filter(row => 
                row.style.display !== 'none'
            );
            
            return visibleRows.map(row => {
                const cells = row.querySelectorAll('td');
                const subjects = Array.from(row.querySelectorAll('.subjects-container .badge')).map(badge => badge.textContent.trim());
                const fullName = cells[1]?.textContent.trim() || '';
                const nameParts = fullName.split(' ');
                const firstname = nameParts[0] || '';
                const lastname = nameParts.slice(1).join(' ') || '';
                
                return {
                    username: cells[4]?.textContent.trim() || '',
                    firstname: firstname,
                    lastname: lastname,
                    password: lastname, // Password is the last name
                    email: cells[2]?.textContent.trim() || '',
                    grade: cells[3]?.textContent.trim() || '',
                    subjects: subjects.join(', '),
                    emailStatus: cells[6]?.textContent.trim() || '',
                    created: cells[7]?.textContent.trim() || ''
                };
            });
        }
        
        function exportToCSV(data, filename) {
            const headers = ['Username', 'First Name', 'Last Name', 'Password', 'Email', 'Grade', 'Subjects', 'Email Status', 'Payment Status', 'Moodle Status', 'Created'];
            const csvContent = [
                headers.join(','),
                ...data.map(row => [
                    `"${row.username}"`,
                    `"${row.firstname}"`,
                    `"${row.lastname}"`,
                    `"${row.password}"`,
                    `"${row.email}"`,
                    `"${row.grade}"`,
                    `"${row.subjects}"`,
                    `"${row.emailStatus}"`,
                    `"${row.paymentStatus || 'Unpaid'}"`,
                    `"${row.moodleStatus || 'active'}"`,
                    `"${row.created}"`
                ].join(','))
            ].join('\n');
            
            downloadFile(csvContent, `${filename}.csv`, 'text/csv');
        }
        
        function exportToExcel(data, filename) {
            // Create HTML table for Excel
            const headers = ['Username', 'First Name', 'Last Name', 'Password', 'Email', 'Grade', 'Subjects', 'Email Status', 'Payment Status', 'Moodle Status', 'Created'];
            let html = '<table border="1">';
            html += '<tr>' + headers.map(h => `<th>${h}</th>`).join('') + '</tr>';
            
            data.forEach(row => {
                html += '<tr>';
                html += `<td>${row.username}</td>`;
                html += `<td>${row.firstname}</td>`;
                html += `<td>${row.lastname}</td>`;
                html += `<td>${row.password}</td>`;
                html += `<td>${row.email}</td>`;
                html += `<td>${row.grade}</td>`;
                html += `<td>${row.subjects}</td>`;
                html += `<td>${row.emailStatus}</td>`;
                html += `<td>${row.paymentStatus || 'Unpaid'}</td>`;
                html += `<td>${row.moodleStatus || 'active'}</td>`;
                html += `<td>${row.created}</td>`;
                html += '</tr>';
            });
            html += '</table>';
            
            downloadFile(html, `${filename}.xls`, 'application/vnd.ms-excel');
        }
        
        function exportToPDF(data, filename) {
            // Create HTML content for PDF
            const headers = ['Username', 'First Name', 'Last Name', 'Password', 'Email', 'Grade', 'Subjects', 'Email Status', 'Payment Status', 'Moodle Status', 'Created'];
            let html = `
                <html>
                <head>
                    <title>Students Export</title>
                    <style>
                        body { font-family: Arial, sans-serif; font-size: 12px; }
                        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
                        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                        th { background-color: #f2f2f2; font-weight: bold; }
                        .header { text-align: center; margin-bottom: 20px; }
                        .date { color: #666; font-size: 10px; }
                    </style>
                </head>
                <body>
                    <div class="header">
                        <h2>Students Export</h2>
                        <div class="date">Generated on: ${new Date().toLocaleString()}</div>
                        <div class="date">Total Records: ${data.length}</div>
                    </div>
                    <table>
                        <thead>
                            <tr>${headers.map(h => `<th>${h}</th>`).join('')}</tr>
                        </thead>
                        <tbody>
            `;
            
            data.forEach(row => {
                html += '<tr>';
                html += `<td>${row.username}</td>`;
                html += `<td>${row.firstname}</td>`;
                html += `<td>${row.lastname}</td>`;
                html += `<td>${row.password}</td>`;
                html += `<td>${row.email}</td>`;
                html += `<td>${row.grade}</td>`;
                html += `<td>${row.subjects}</td>`;
                html += `<td>${row.emailStatus}</td>`;
                html += `<td>${row.paymentStatus || 'Unpaid'}</td>`;
                html += `<td>${row.moodleStatus || 'active'}</td>`;
                html += `<td>${row.created}</td>`;
                html += '</tr>';
            });
            
            html += `
                        </tbody>
                    </table>
                </body>
                </html>
            `;
            
            // Open in new window for printing/saving as PDF
            const printWindow = window.open('', '_blank');
            printWindow.document.write(html);
            printWindow.document.close();
            printWindow.print();
        }
        
        function exportToHTML(data, filename) {
            const headers = ['Username', 'First Name', 'Last Name', 'Password', 'Email', 'Grade', 'Subjects', 'Email Status', 'Payment Status', 'Moodle Status', 'Created'];
            let html = `
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Students Export</title>
                    <style>
                        body { font-family: Arial, sans-serif; margin: 20px; }
                        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
                        th, td { border: 1px solid #ddd; padding: 12px; text-align: left; }
                        th { background-color: #f8f9fa; font-weight: bold; }
                        .header { text-align: center; margin-bottom: 30px; }
                        .date { color: #666; font-size: 14px; margin: 10px 0; }
                        .count { color: #007bff; font-weight: bold; }
                    </style>
                </head>
                <body>
                    <div class="header">
                        <h1>Students Export</h1>
                        <div class="date">Generated on: ${new Date().toLocaleString()}</div>
                        <div class="date">Total Records: <span class="count">${data.length}</span></div>
                    </div>
                    <table>
                        <thead>
                            <tr>${headers.map(h => `<th>${h}</th>`).join('')}</tr>
                        </thead>
                        <tbody>
            `;
            
            data.forEach(row => {
                html += '<tr>';
                html += `<td>${row.username}</td>`;
                html += `<td>${row.firstname}</td>`;
                html += `<td>${row.lastname}</td>`;
                html += `<td>${row.password}</td>`;
                html += `<td>${row.email}</td>`;
                html += `<td>${row.grade}</td>`;
                html += `<td>${row.subjects}</td>`;
                html += `<td>${row.emailStatus}</td>`;
                html += `<td>${row.paymentStatus || 'Unpaid'}</td>`;
                html += `<td>${row.moodleStatus || 'active'}</td>`;
                html += `<td>${row.created}</td>`;
                html += '</tr>';
            });
            
            html += `
                        </tbody>
                    </table>
                </body>
                </html>
            `;
            
            downloadFile(html, `${filename}.html`, 'text/html');
        }
        
        function printTable() {
            // Get all filtered data from server instead of just visible rows
            getAllFilteredStudents().then(allStudents => {
                const headers = ['Username', 'First Name', 'Last Name', 'Password', 'Email', 'Grade', 'Subjects', 'Email Status', 'Payment Status', 'Moodle Status', 'Created'];
                
                let html = `
                    <html>
                    <head>
                        <title>Students Print</title>
                        <style>
                            body { font-family: Arial, sans-serif; font-size: 12px; margin: 0; padding: 20px; }
                            table { width: 100%; border-collapse: collapse; }
                            th, td { border: 1px solid #000; padding: 6px; text-align: left; font-size: 10px; }
                            th { background-color: #f0f0f0; font-weight: bold; }
                            .header { text-align: center; margin-bottom: 20px; }
                            .date { font-size: 10px; color: #666; }
                            .page-break { page-break-before: always; }
                            @media print {
                                body { margin: 0; }
                                .no-print { display: none; }
                            }
                        </style>
                    </head>
                    <body>
                        <div class="header">
                            <h2>Students Report</h2>
                            <div class="date">Generated: ${new Date().toLocaleString()}</div>
                            <div class="date">Total Records: ${allStudents.length}</div>
                        </div>
                        <table>
                            <thead>
                                <tr>${headers.map(h => `<th>${h}</th>`).join('')}</tr>
                            </thead>
                            <tbody>
                `;
                
                allStudents.forEach((row, index) => {
                    // Add page break every 50 rows to prevent overly long pages
                    if (index > 0 && index % 50 === 0) {
                        html += '</tbody></table><div class="page-break"></div><table><thead><tr>' + headers.map(h => `<th>${h}</th>`).join('') + '</tr></thead><tbody>';
                    }
                    
                    html += '<tr>';
                    html += `<td>${row.username}</td>`;
                    html += `<td>${row.firstname}</td>`;
                    html += `<td>${row.lastname}</td>`;
                    html += `<td>${row.password}</td>`;
                    html += `<td>${row.email}</td>`;
                    html += `<td>${row.grade}</td>`;
                    html += `<td>${row.subjects}</td>`;
                    html += `<td>${row.emailStatus}</td>`;
                    html += `<td>${row.created}</td>`;
                    html += '</tr>';
                });
                
                html += `
                            </tbody>
                        </table>
                    </body>
                    </html>
                `;
                
                const printWindow = window.open('', '_blank');
                printWindow.document.write(html);
                printWindow.document.close();
                printWindow.print();
            }).catch(error => {
                console.error('Error getting filtered students:', error);
                alert('Error loading data for printing. Please try again.');
            });
        }
        
        function getAllFilteredStudents() {
            // Get current filter values
            const form = document.querySelector('form[method="get"]');
            const formData = new FormData(form);
            
            // Remove pagination parameters
            formData.delete('page');
            
            // Build query string
            const params = new URLSearchParams();
            for (let [key, value] of formData.entries()) {
                if (value) {
                    params.append(key, value);
                }
            }
            
            // Fetch all filtered data from server
            return fetch(`dashboard.php?${params.toString()}&export_all=1`, {
                headers: { 'Accept': 'application/json' }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success && data.students) {
                    return data.students.map(student => {
                        const subjects = student.subjects || [];
                        const subjectNames = subjects.map(subject => subject.name || subject);
                        
                        return {
                            username: student.username || '',
                            firstname: student.first_name || '',
                            lastname: student.last_name || '',
                            password: student.last_name || '', // Password is the last name
                            email: student.email || '',
                            grade: student.grade_name || '',
                            subjects: subjectNames.join(', '),
                            emailStatus: student.email_verified ? 'Verified' : 'Pending',
                            paymentStatus: student.payment_status || 'Unpaid',
                            moodleStatus: student.moodle_status || 'active',
                            created: student.created_at || ''
                        };
                    });
                } else {
                    throw new Error('Failed to fetch filtered students');
                }
            });
        }
        
        function downloadFile(content, filename, mimeType) {
            const blob = new Blob([content], { type: mimeType });
            const url = window.URL.createObjectURL(blob);
            const link = document.createElement('a');
            link.href = url;
            link.download = filename;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            window.URL.revokeObjectURL(url);
        }

        // Bot Monitoring Functions
        let botStatusInterval;
        
        // Initialize bot monitoring on page load
        document.addEventListener('DOMContentLoaded', function() {
            refreshBotStatus();
            refreshBotMetrics();
            refreshLogs();
            
            // Auto-refresh every 30 seconds
            botStatusInterval = setInterval(() => {
                refreshBotStatus();
                refreshBotMetrics();
            }, 30000);
        });
        
        async function refreshBotStatus() {
            try {
                const response = await fetch('bot_monitor_api.php?action=status');
                const data = await response.json();
                
                // Update status badge
                const statusBadge = document.getElementById('botStatus');
                if (data.status === 'running') {
                    statusBadge.className = 'badge bg-success';
                    statusBadge.textContent = 'Running';
                    document.getElementById('triggerBotBtn').disabled = true;
                } else if (data.status === 'idle') {
                    statusBadge.className = 'badge bg-secondary';
                    statusBadge.textContent = 'Idle';
                    document.getElementById('triggerBotBtn').disabled = false;
                } else {
                    statusBadge.className = 'badge bg-warning';
                    statusBadge.textContent = 'Unknown';
                }
                
                // Update other status info
                document.getElementById('lastRun').textContent = data.last_run || 'Never';
                document.getElementById('errorCount').textContent = data.error_count || '0';
                document.getElementById('logSize').textContent = formatFileSize(data.log_size || 0);
                
            } catch (error) {
                console.error('Error refreshing bot status:', error);
                document.getElementById('botStatus').textContent = 'Error';
                document.getElementById('botStatus').className = 'badge bg-danger';
            }
        }
        
        async function refreshBotMetrics() {
            try {
                const response = await fetch('bot_monitor_api.php?action=metrics');
                const data = await response.json();
                
                document.getElementById('totalSyncs').textContent = data.total_syncs || '0';
                document.getElementById('successfulSyncs').textContent = data.successful_syncs || '0';
                document.getElementById('failedSyncs').textContent = data.failed_syncs || '0';
                document.getElementById('usersUploaded').textContent = data.total_users_uploaded || '0';
                document.getElementById('enrollments').textContent = data.total_enrollments || '0';
                document.getElementById('avgSyncTime').textContent = (data.average_sync_time || '0') + 's';
                
            } catch (error) {
                console.error('Error refreshing bot metrics:', error);
            }
        }
        
        async function refreshLogs() {
            try {
                const response = await fetch('bot_monitor_api.php?action=logs');
                const data = await response.json();
                
                const logsContainer = document.getElementById('botLogs');
                if (data.logs && data.logs.length > 0) {
                    logsContainer.innerHTML = data.logs.map(log => {
                        const levelClass = log.level === 'error' ? 'text-danger' : 
                                          log.level === 'warning' ? 'text-warning' : 'text-light';
                        return `<div class="${levelClass}">[${log.timestamp || 'N/A'}] ${log.message}</div>`;
                    }).join('');
                } else {
                    logsContainer.innerHTML = '<div class="text-muted text-center">No logs available</div>';
                }
                
            } catch (error) {
                console.error('Error refreshing logs:', error);
                document.getElementById('botLogs').innerHTML = '<div class="text-danger">Error loading logs</div>';
            }
        }
        
        async function triggerBot() {
            const btn = document.getElementById('triggerBotBtn');
            const originalText = btn.innerHTML;
            
            try {
                btn.disabled = true;
                btn.innerHTML = '<i class="bi bi-hourglass-split me-1"></i>Starting...';
                
                const response = await fetch('bot_monitor_api.php?action=trigger');
                const data = await response.json();
                
                if (data.success) {
                    showModalAlert('success', 'Bot triggered successfully! Check the logs for progress.');
                    // Refresh status after a short delay
                    setTimeout(() => {
                        refreshBotStatus();
                        refreshLogs();
                    }, 2000);
                } else {
                    showModalAlert('warning', data.message || 'Failed to trigger bot');
                }
                
            } catch (error) {
                console.error('Error triggering bot:', error);
                showModalAlert('danger', 'Error triggering bot: ' + error.message);
            } finally {
                btn.disabled = false;
                btn.innerHTML = originalText;
            }
        }
        
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 B';
            const k = 1024;
            const sizes = ['B', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i];
        }
        
        // Clean up interval on page unload
        window.addEventListener('beforeunload', function() {
            if (botStatusInterval) {
                clearInterval(botStatusInterval);
            }
        });

        // Status toggle handlers
        document.addEventListener('click', function(e) {
            if (e.target.closest('.toggle-email-status')) {
                e.preventDefault();
                const button = e.target.closest('.toggle-email-status');
                const studentId = button.dataset.studentId;
                const currentStatus = button.dataset.currentStatus;
                const newStatus = currentStatus === '1' ? '0' : '1';
                
                if (confirm(`Are you sure you want to ${newStatus === '1' ? 'verify' : 'unverify'} this student's email?`)) {
                    updateEmailStatus(studentId, newStatus, button);
                }
            }
            
            if (e.target.closest('.toggle-payment-status')) {
                e.preventDefault();
                const button = e.target.closest('.toggle-payment-status');
                const studentId = button.dataset.studentId;
                const currentStatus = button.dataset.currentStatus;
                const newStatus = currentStatus === 'paid' ? 'unpaid' : 'paid';
                
                if (confirm(`Are you sure you want to mark this student as ${newStatus}?`)) {
                    updatePaymentStatus(studentId, newStatus, button);
                }
            }
        });

        function updateEmailStatus(studentId, newStatus, button) {
            const formData = new FormData();
            formData.append('student_id', studentId);
            formData.append('status_type', 'email');
            formData.append('new_status', newStatus);
            formData.append('csrf_token', '<?php echo generateCSRFToken(); ?>');
            
            fetch('update_status.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update the UI
                    const badge = button.parentElement.querySelector('.badge');
                    const icon = badge.querySelector('i');
                    const text = badge.querySelector('i').nextSibling;
                    
                    if (newStatus === '1') {
                        badge.className = 'badge bg-success';
                        icon.className = 'bi bi-check-circle me-1';
                        text.textContent = 'Verified';
                    } else {
                        badge.className = 'badge bg-warning';
                        icon.className = 'bi bi-exclamation-triangle me-1';
                        text.textContent = 'Pending';
                    }
                    
                    button.dataset.currentStatus = newStatus;
                    showToast('success', data.message);
                } else {
                    showToast('error', data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showToast('error', 'Failed to update email status');
            });
        }

        function updatePaymentStatus(studentId, newStatus, button) {
            const formData = new FormData();
            formData.append('student_id', studentId);
            formData.append('status_type', 'payment');
            formData.append('new_status', newStatus);
            formData.append('csrf_token', '<?php echo generateCSRFToken(); ?>');
            
            fetch('update_status.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update the UI
                    const badge = button.parentElement.querySelector('.badge');
                    const icon = badge.querySelector('i');
                    const text = badge.querySelector('i').nextSibling;
                    
                    if (newStatus === 'paid') {
                        badge.className = 'badge bg-success';
                        icon.className = 'bi bi-credit-card me-1';
                        text.textContent = 'Paid';
                    } else {
                        badge.className = 'badge bg-danger';
                        icon.className = 'bi bi-x-circle me-1';
                        text.textContent = 'Unpaid';
                    }
                    
                    button.dataset.currentStatus = newStatus;
                    if (newStatus === 'paid') {
                        showToast('success', data.message + ' Bot will sync enrollment to Moodle automatically.');
                    } else {
                        showToast('success', data.message);
                    }
                } else {
                    showToast('error', data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showToast('error', 'Failed to update payment status');
            });
        }

        function showToast(type, message) {
            // Create toast element
            const toast = document.createElement('div');
            toast.className = `alert alert-${type === 'success' ? 'success' : 'danger'} alert-dismissible fade show position-fixed`;
            toast.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
            toast.innerHTML = `
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            
            document.body.appendChild(toast);
            
            // Auto remove after 3 seconds
            setTimeout(() => {
                if (toast.parentNode) {
                    toast.parentNode.removeChild(toast);
                }
            }, 3000);
        }
    </script>

    <!-- Professional Modern Footer -->
    <footer class="footer-modern">
        <div class="container-fluid">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <div class="d-flex align-items-center">
                        <div class="footer-logo me-3">
                            <div class="logo-icon">
                                <i class="bi bi-mortarboard-fill"></i>
                            </div>
                        </div>
                        <div class="footer-brand">
                            <h6 class="brand-title"><?php echo APP_NAME; ?> Learning Platform</h6>
                            <p class="brand-subtitle">Advanced Student Management System</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-6 text-md-end">
                    <div class="d-flex align-items-center justify-content-md-end">
                        <div class="developer-info">
                            <div class="d-flex align-items-center">
                                <div class="developer-logo me-2">
                                    <i class="bi bi-code-slash"></i>
                                </div>
                                <div class="developer-text">
                                    <span class="powered-by">Powered by</span>
                                    <a href="https://binarygroup.com" target="_blank" class="developer-link">
                                        BinaryGroup.com
                                    </a>
                                    <span class="developer-tagline">Professional Web Development</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="footer-divider"></div>
            <div class="row align-items-center">
                <div class="col-md-6">
                    <div class="footer-info">
                        <i class="bi bi-shield-check me-2"></i>
                        <span>Secure & Reliable Student Management</span>
                    </div>
                </div>
                <div class="col-md-6 text-md-end">
                    <div class="footer-copyright">
                        <i class="bi bi-calendar3 me-2"></i>
                        <span>© <?php echo date('Y'); ?> All rights reserved</span>
                    </div>
                </div>
            </div>
        </div>
    </footer>

    <style>
        .footer-modern {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 50%, #0f3460 100%);
            color: #ffffff;
            padding: 1.5rem 0;
            margin-top: 2rem;
            border-top: 3px solid #007bff;
            position: relative;
            overflow: hidden;
        }
        
        .footer-modern::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 2px;
            background: linear-gradient(90deg, #007bff, #28a745, #ffc107, #dc3545, #6f42c1);
            animation: gradientShift 3s ease-in-out infinite;
        }
        
        @keyframes gradientShift {
            0%, 100% { opacity: 0.8; }
            50% { opacity: 1; }
        }
        
        .footer-logo .logo-icon {
            width: 40px;
            height: 40px;
            background: linear-gradient(135deg, #007bff, #0056b3);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 4px 15px rgba(0, 123, 255, 0.3);
            transition: all 0.3s ease;
        }
        
        .footer-logo .logo-icon:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0, 123, 255, 0.4);
        }
        
        .footer-logo .logo-icon i {
            font-size: 18px;
            color: #ffffff;
        }
        
        .footer-brand .brand-title {
            color: #ffffff;
            font-size: 16px;
            font-weight: 600;
            margin: 0;
            letter-spacing: 0.5px;
        }
        
        .footer-brand .brand-subtitle {
            color: #b8c5d1;
            font-size: 12px;
            margin: 0;
            font-weight: 400;
        }
        
        .developer-logo {
            width: 32px;
            height: 32px;
            background: linear-gradient(135deg, #28a745, #20c997);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 3px 10px rgba(40, 167, 69, 0.3);
            transition: all 0.3s ease;
        }
        
        .developer-logo:hover {
            transform: translateY(-1px);
            box-shadow: 0 5px 15px rgba(40, 167, 69, 0.4);
        }
        
        .developer-logo i {
            font-size: 14px;
            color: #ffffff;
        }
        
        .developer-text {
            display: flex;
            flex-direction: column;
            align-items: flex-end;
        }
        
        .powered-by {
            color: #b8c5d1;
            font-size: 10px;
            font-weight: 500;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .developer-link {
            color: #ffffff;
            text-decoration: none;
            font-size: 14px;
            font-weight: 600;
            transition: all 0.3s ease;
            position: relative;
        }
        
        .developer-link:hover {
            color: #28a745;
            transform: translateX(2px);
        }
        
        .developer-link::after {
            content: '';
            position: absolute;
            bottom: -2px;
            left: 0;
            width: 0;
            height: 2px;
            background: linear-gradient(90deg, #28a745, #20c997);
            transition: width 0.3s ease;
        }
        
        .developer-link:hover::after {
            width: 100%;
        }
        
        .developer-tagline {
            color: #b8c5d1;
            font-size: 10px;
            font-weight: 400;
        }
        
        .footer-divider {
            height: 1px;
            background: linear-gradient(90deg, transparent, #007bff, transparent);
            margin: 1rem 0;
            opacity: 0.6;
        }
        
        .footer-info, .footer-copyright {
            color: #b8c5d1;
            font-size: 11px;
            font-weight: 500;
            display: flex;
            align-items: center;
        }
        
        .footer-info i, .footer-copyright i {
            color: #007bff;
            font-size: 12px;
        }
        
        .footer-modern .row {
            margin: 0;
        }
        
        .footer-modern .col-md-6 {
            padding: 0.5rem 0;
        }
        
        @media (max-width: 768px) {
            .footer-modern {
                padding: 1rem 0;
            }
            
            .footer-brand .brand-title {
                font-size: 14px;
            }
            
            .footer-brand .brand-subtitle {
                font-size: 11px;
            }
            
            .developer-text {
                align-items: flex-start;
            }
            
            .footer-info, .footer-copyright {
                font-size: 10px;
                margin-top: 0.5rem;
            }
        }
    </style>

    <!-- Withdrawal Modal -->
    <div class="modal fade" id="withdrawModal" tabindex="-1" aria-labelledby="withdrawModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="withdrawModalLabel">Withdraw Money</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form id="withdrawalForm">
                        <div class="alert alert-info">
                            <strong>Available Balance:</strong> SZL <?php echo number_format($wallet['balance'] ?? 0, 2); ?>
                        </div>
                        
                        <div class="mb-3">
                            <label for="withdrawAmount" class="form-label">Withdrawal Amount (SZL)</label>
                            <input type="number" 
                                   class="form-control" 
                                   id="withdrawAmount" 
                                   name="amount" 
                                   min="<?php echo MIN_WITHDRAWAL_AMOUNT; ?>" 
                                   max="<?php echo $wallet['balance'] ?? 0; ?>" 
                                   step="0.01" 
                                   required>
                            <small class="form-text text-muted">
                                Minimum: SZL <?php echo MIN_WITHDRAWAL_AMOUNT; ?>
                            </small>
                        </div>

                        <div class="mb-3">
                            <label for="momoNumber" class="form-label">MTN MoMo Number</label>
                            <input type="text" 
                                   class="form-control" 
                                   id="momoNumber" 
                                   name="momo_number" 
                                   placeholder="268XXXXXXXX or 7XXXXXXX"
                                   pattern="(268[0-9]{8}|7[68][0-9]{7})"
                                   value="<?php echo htmlspecialchars($wallet['momo_number'] ?? ''); ?>"
                                   required>
                            <small class="form-text text-muted">
                                Eswatini format: 268XXXXXXXX or 76XXXXXXX/78XXXXXXX
                            </small>
                        </div>

                        <div class="alert alert-warning">
                            <i class="bi bi-exclamation-triangle me-2"></i>
                            <strong>Important:</strong> Funds will be transferred to your MoMo account within a few minutes.
                        </div>

                        <div id="withdrawalError" class="alert alert-danger d-none"></div>
                        <div id="withdrawalSuccess" class="alert alert-success d-none"></div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" id="submitWithdrawal">
                        <i class="bi bi-cash-coin me-1"></i> Withdraw Money
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script>
    // Withdrawal form submission
    document.getElementById('submitWithdrawal')?.addEventListener('click', async function() {
        const form = document.getElementById('withdrawalForm');
        const amount = document.getElementById('withdrawAmount').value;
        const momoNumber = document.getElementById('momoNumber').value;
        const errorDiv = document.getElementById('withdrawalError');
        const successDiv = document.getElementById('withdrawalSuccess');
        const submitBtn = this;

        // Validate form
        if (!form.checkValidity()) {
            form.reportValidity();
            return;
        }

        // Disable button and show loading
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-1"></span> Processing...';
        errorDiv.classList.add('d-none');
        successDiv.classList.add('d-none');

        try {
            const response = await fetch('request_withdrawal.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `amount=${amount}&momo_number=${momoNumber}`
            });

            const result = await response.json();

            if (result.success) {
                successDiv.textContent = result.message || 'Withdrawal request submitted successfully!';
                successDiv.classList.remove('d-none');
                
                // Reload page after 2 seconds to show updated balance
                setTimeout(() => {
                    window.location.reload();
                }, 2000);
            } else {
                errorDiv.textContent = result.error || 'Withdrawal failed. Please try again.';
                errorDiv.classList.remove('d-none');
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="bi bi-cash-coin me-1"></i> Withdraw Money';
            }
        } catch (error) {
            errorDiv.textContent = 'Network error. Please check your connection and try again.';
            errorDiv.classList.remove('d-none');
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<i class="bi bi-cash-coin me-1"></i> Withdraw Money';
        }
    });
    </script>

</body>
</html>


