<?php
require_once '../includes/functions.php';
require_once '../includes/school_context.php';
requireAdminLogin();
requireSchool(); // Ensure school is selected

// Get current school context
$currentSchool = getCurrentSchool();

$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $firstName = trim($_POST['first_name'] ?? '');
    $lastName = trim($_POST['last_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $gradeId = (int)($_POST['grade_id'] ?? 0);
    $password = (string)($_POST['password'] ?? '');
    $confirm = (string)($_POST['password_confirm'] ?? '');
    $subjectIds = array_filter(array_map('intval', (array)($_POST['subject_ids'] ?? [])));

    if ($firstName === '') $errors[] = 'First name is required.';
    if ($lastName === '') $errors[] = 'Last name is required.';
    if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = 'Valid email is required.';
    if ($gradeId <= 0) $errors[] = 'Please select a grade.';
    if ($password === '' || strlen($password) < 8) $errors[] = 'Password must be at least 8 characters.';
    if ($password !== $confirm) $errors[] = 'Passwords do not match.';
    
    // Debug logging
    if (defined('APP_DEBUG') && APP_DEBUG) {
        error_log('Admin create form validation - First: ' . $firstName . ', Last: ' . $lastName . ', Email: ' . $email . ', Grade: ' . $gradeId . ', Password length: ' . strlen($password));
    }
    
    // Unique email
    if (empty($errors)) {
        $pdo = getDB();
        $stmt = $pdo->prepare('SELECT id FROM students WHERE email = ? AND academy_reference = ?');
        $stmt->execute([$email, $currentSchool]);
        if ($stmt->fetch()) {
            $errors[] = 'Email address is already registered.';
        }
    }
    
    if (empty($errors)) {
        try {
            createStudent([
                'first_name' => $firstName,
                'last_name' => $lastName,
                'email' => $email,
                'grade_id' => $gradeId,
                'password_plain' => $password,
                'subject_ids' => $subjectIds,
                'academy_reference' => $currentSchool // Add school context
            ]);
            $success = true;
        } catch (Throwable $e) {
            $errors[] = 'Error creating student: ' . $e->getMessage();
            // Log the full error for debugging
            error_log('Admin create student error: ' . $e->getMessage() . ' - Trace: ' . $e->getTraceAsString());
        }
    }
}

$grades = getGrades();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Student - <?php echo APP_NAME; ?></title>
    
    <!-- Favicon -->
    <link rel="icon" type="image/svg+xml" href="../images/img-melanesys-logo.svg">
    <link rel="alternate icon" href="../images/img-melanesys-logo.svg">
    <link rel="apple-touch-icon" href="../images/img-melanesys-logo.svg">
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <style>
        :root{
            --brand-primary:#6f42c1;
            --brand-accent:#20c997;
            --brand-soft:#f3f0ff;
        }
        body{background:linear-gradient(135deg,var(--brand-soft),#e8fff9);} 
        .hero{
            background: radial-gradient(1000px 400px at 10% 0%, rgba(111,66,193,.15), rgba(32,201,151,.1)),
                linear-gradient(135deg, rgba(111,66,193,.12), rgba(32,201,151,.12));
            border-radius: 1rem;
        }
        .brand-badge{background:var(--brand-primary);}    
        .brand-underline{position:relative; display:inline-block}
        .brand-underline::after{content:""; position:absolute; left:0; right:0; bottom:-.25rem; height:.35rem; background:linear-gradient(90deg,var(--brand-primary),var(--brand-accent)); border-radius:1rem}
        .card-modern{border:0; box-shadow:0 1rem 2rem rgba(0,0,0,.07);}        
        .input-icon{position:relative}
        .input-icon .bi{position:absolute; left:.9rem; top:50%; transform:translateY(-50%); color:#6c757d}
        .input-icon input, .input-icon select{padding-left:2.4rem}
        .list-gradient .list-group-item{border:0; background:linear-gradient(90deg,#ffffff,#f8f9fa)}
        .btn-gradient{background:linear-gradient(90deg,var(--brand-primary),var(--brand-accent)); border:0}
        .btn-gradient:hover{filter:brightness(.95)}
        .form-check-input:checked{background-color:var(--brand-primary); border-color:var(--brand-primary)}
        .form-check-label{cursor:pointer; user-select:none}
        .subject-checkbox:hover{background-color:rgba(111,66,193,.05); border-radius:.375rem}
    </style>
</head>
<body class="bg-light">
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container-fluid">
            <a class="navbar-brand" href="dashboard.php">
                <i class="bi bi-speedometer2 me-2"></i>Admin Dashboard
            </a>
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="dashboard.php">
                    <i class="bi bi-arrow-left me-1"></i>Back to Dashboard
                </a>
                <a class="nav-link" href="logout.php">
                    <i class="bi bi-box-arrow-right me-1"></i>Logout
                </a>
            </div>
        </div>
    </nav>

    <div class="container my-4">
        <div class="hero p-4 p-md-5 mb-4 text-dark">
            <div class="d-flex align-items-center mb-2">
                <span class="badge brand-badge text-white me-2">Create</span>
                <small class="text-muted">Add a student quickly</small>
            </div>
            <h1 class="display-6 fw-bold mb-2 brand-underline">New Student</h1>
            <p class="mb-0">Enter the student's details, set a password, pick subjects, and assign a grade.</p>
        </div>

        <div class="row justify-content-center">
            <div class="col-lg-10">
                <div class="card card-modern">
                    <div class="card-body p-4 p-md-5">
                        <?php if ($success): ?>
                            <div class="alert alert-success">
                                <i class="bi bi-check-circle me-1"></i>Student created and credentials emailed.
                            </div>
                        <?php endif; ?>

                        <?php if (!empty($errors)): ?>
                            <div class="alert alert-danger">
                                <h6><i class="bi bi-exclamation-triangle me-1"></i>Please correct the following errors:</h6>
                                <ul class="mb-0">
                                    <?php foreach ($errors as $error): ?>
                                        <li><?php echo htmlspecialchars($error); ?></li>
                                    <?php endforeach; ?>
                                </ul>
                                <?php if (defined('APP_DEBUG') && APP_DEBUG): ?>
                                    <hr>
                                    <small class="text-muted">
                                        <strong>Debug Info:</strong><br>
                                        Form Data: <?php echo htmlspecialchars(json_encode($_POST)); ?><br>
                                        Request Method: <?php echo htmlspecialchars($_SERVER['REQUEST_METHOD'] ?? 'Not set'); ?>
                                    </small>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>

                        <form method="POST" action="create.php">
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <label class="form-label">First Name *</label>
                                    <div class="input-icon">
                                        <i class="bi bi-person"></i>
                                        <input class="form-control" name="first_name" placeholder="e.g. Jane" required />
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Last Name *</label>
                                    <div class="input-icon">
                                        <i class="bi bi-person"></i>
                                        <input class="form-control" name="last_name" placeholder="e.g. Doe" required />
                                    </div>
                                </div>
                                <div class="col-md-8">
                                    <label class="form-label">Email *</label>
                                    <div class="input-icon">
                                        <i class="bi bi-envelope"></i>
                                        <input type="email" class="form-control" name="email" placeholder="you@example.com" required />
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <label class="form-label">Grade *</label>
                                    <div class="input-icon">
                                        <i class="bi bi-mortarboard"></i>
                                        <select class="form-select" name="grade_id" id="gradeSelect" required>
                                            <option value="">Select grade</option>
                                            <?php foreach ($grades as $grade): ?>
                                                <option value="<?php echo $grade['id']; ?>"><?php echo htmlspecialchars($grade['name']); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Create Password *</label>
                                    <div class="input-icon">
                                        <i class="bi bi-shield-lock"></i>
                                        <input type="password" class="form-control" name="password" minlength="8" placeholder="At least 8 characters" required />
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Confirm Password *</label>
                                    <div class="input-icon">
                                        <i class="bi bi-shield-check"></i>
                                        <input type="password" class="form-control" name="password_confirm" minlength="8" placeholder="Re-type password" required />
                                    </div>
                                </div>
                                <div class="col-12">
                                    <label class="form-label">Subjects (choose one or more)</label>
                                    <div id="subjectsContainer" class="border rounded p-3 bg-light" style="min-height: 120px;">
                                        <div class="text-muted text-center py-4">
                                            <i class="bi bi-mortarboard fs-1 d-block mb-2"></i>
                                            Please select a grade first to see available subjects
                                        </div>
                                    </div>
                                </div>
                                <div class="col-12 d-grid d-sm-flex gap-2 mt-2">
                                    <button type="submit" class="btn btn-gradient text-white px-4">
                                        <i class="bi bi-save me-1"></i>Create
                                    </button>
                                    <a href="dashboard.php" class="btn btn-outline-secondary px-4">
                                        <i class="bi bi-x-lg me-1"></i>Cancel
                                    </a>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
    const gradeSelect = document.getElementById('gradeSelect');
    const subjectsContainer = document.getElementById('subjectsContainer');
    
    function renderSubjects(items){
        
        // Update the checkbox container
        subjectsContainer.innerHTML = '';
        if(!items || !items.length){
            subjectsContainer.innerHTML = `
                <div class="text-muted text-center py-4">
                    <i class="bi bi-exclamation-circle fs-1 d-block mb-2"></i>
                    No subjects available for this grade yet.
                </div>
            `;
            return;
        }
        
        // Create header with Select All / Clear All buttons
        const buttonRow = document.createElement('div');
        buttonRow.className = 'row mb-3';
        buttonRow.innerHTML = `
            <div class="col-12">
                <div class="d-flex gap-2">
                    <button type="button" class="btn btn-sm btn-outline-primary" id="selectAllSubjects">
                        <i class="bi bi-check-square me-1"></i>Select All
                    </button>
                    <button type="button" class="btn btn-sm btn-outline-secondary" id="clearAllSubjects">
                        <i class="bi bi-square me-1"></i>Clear All
                    </button>
                </div>
            </div>
        `;
        subjectsContainer.appendChild(buttonRow);
        
        // Create checkbox grid
        const row = document.createElement('div');
        row.className = 'row g-2';
        
        items.forEach(s => {
            const col = document.createElement('div');
            col.className = 'col-md-6 col-lg-4';
            
            const checkboxDiv = document.createElement('div');
            checkboxDiv.className = 'form-check subject-checkbox p-2';
            
            const checkbox = document.createElement('input');
            checkbox.type = 'checkbox';
            checkbox.className = 'form-check-input';
            checkbox.name = 'subject_ids[]';
            checkbox.value = s.id;
            checkbox.id = 'subject_' + s.id;
            
            const label = document.createElement('label');
            label.className = 'form-check-label w-100';
            label.htmlFor = 'subject_' + s.id;
            label.textContent = s.name + (s.code ? ' ('+s.code+')' : '');
            
            checkboxDiv.appendChild(checkbox);
            checkboxDiv.appendChild(label);
            col.appendChild(checkboxDiv);
            row.appendChild(col);
        });
        
        subjectsContainer.appendChild(row);
        
        // Add event listeners for Select All / Clear All buttons
        document.getElementById('selectAllSubjects').addEventListener('click', function() {
            const checkboxes = subjectsContainer.querySelectorAll('input[name="subject_ids[]"]');
            checkboxes.forEach(cb => cb.checked = true);
        });
        
        document.getElementById('clearAllSubjects').addEventListener('click', function() {
            const checkboxes = subjectsContainer.querySelectorAll('input[name="subject_ids[]"]');
            checkboxes.forEach(cb => cb.checked = false);
        });
    }
    
    async function loadSubjects(){
        const gid = gradeSelect.value;
        renderSubjects([]);
        if(!gid) return;
        
        // Show loading message
        subjectsContainer.innerHTML = `
            <div class="text-muted text-center py-4">
                <i class="bi bi-hourglass-split fs-1 d-block mb-2"></i>
                Loading subjects...
            </div>
        `;
        
        try{
            const res = await fetch('../register.php?action=subjects&grade_id='+encodeURIComponent(gid), {headers: {'Accept':'application/json'}});
            if(!res.ok) {
                throw new Error('Server error: ' + res.status);
            }
            const data = await res.json();
            renderSubjects(data || []);
        }catch(e){
            console.error('Error loading subjects:', e);
            // Fallback: show sample subjects when database is not available
            const sampleSubjects = [
                {id: 1, name: 'Mathematics', code: 'MATH'},
                {id: 2, name: 'English', code: 'ENG'},
                {id: 3, name: 'Science', code: 'SCI'},
                {id: 4, name: 'History', code: 'HIST'},
                {id: 5, name: 'Geography', code: 'GEO'},
                {id: 6, name: 'Art', code: 'ART'},
                {id: 7, name: 'Physical Education', code: 'PE'},
                {id: 8, name: 'Computer Studies', code: 'CS'}
            ];
            
            subjectsContainer.innerHTML = `
                <div class="alert alert-warning mb-3">
                    <i class="bi bi-exclamation-triangle me-1"></i>
                    Database connection issue. Showing sample subjects for testing.
                </div>
            `;
            renderSubjects(sampleSubjects);
        }
    }
    
    gradeSelect?.addEventListener('change', loadSubjects);
    
    // Debug logging
    console.log('Admin create page loaded');
    console.log('Grade select element:', gradeSelect);
    console.log('Subjects container:', subjectsContainer);
    
    // Test if elements exist
    if (!gradeSelect) {
        console.error('Grade select element not found!');
    }
    if (!subjectsContainer) {
        console.error('Subjects container element not found!');
    }
    </script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
