<?php
/**
 * Tenant File Upload - Usage Examples
 * 
 * Demonstrates how to use the TenantFileUpload class
 */

require_once 'config.php';
require_once 'includes/tenant_directory_manager.php';
require_once 'includes/tenant_file_upload.php';

// Create database connection
$conn = new PDO(
    "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME,
    DB_USER,
    DB_PASS
);

// Initialize upload handler
$dirManager = new TenantDirectoryManager($conn);
$uploader = new TenantFileUpload($conn, $dirManager);

// =====================================================
// EXAMPLE 1: Upload Document
// =====================================================

echo "=== Example 1: Upload Document ===\n\n";

if (isset($_FILES['document'])) {
    $result = $uploader->uploadDocument(
        $_SESSION['academy_reference'],
        $_FILES['document'],
        [
            'description' => 'Student handbook 2025',
            'tags' => 'handbook,rules,2025',
            'related_id' => null,
            'related_type' => null
        ]
    );
    
    if ($result['success']) {
        echo "✅ Document uploaded successfully!\n";
        echo "File ID: {$result['file_id']}\n";
        echo "Filename: {$result['filename']}\n";
        echo "URL: {$result['url']}\n";
    } else {
        echo "❌ Upload failed: {$result['error']}\n";
    }
}

// HTML Form for document upload
?>
<form method="post" enctype="multipart/form-data">
    <h3>Upload Document</h3>
    <input type="file" name="document" accept=".pdf,.doc,.docx,.xls,.xlsx" required>
    <input type="text" name="description" placeholder="Document description">
    <button type="submit">Upload Document</button>
</form>
<?php

// =====================================================
// EXAMPLE 2: Upload Profile Photo
// =====================================================

echo "\n=== Example 2: Upload Profile Photo ===\n\n";

if (isset($_FILES['profile_photo'])) {
    $student_id = $_POST['student_id'] ?? null;
    
    if ($student_id) {
        $result = $uploader->uploadProfilePhoto(
            $_SESSION['academy_reference'],
            $student_id,
            $_FILES['profile_photo']
        );
        
        if ($result['success']) {
            echo "✅ Profile photo uploaded!\n";
            echo "File ID: {$result['file_id']}\n";
            echo "URL: {$result['url']}\n";
        } else {
            echo "❌ Upload failed: {$result['error']}\n";
        }
    }
}

// HTML Form for photo upload
?>
<form method="post" enctype="multipart/form-data">
    <h3>Upload Profile Photo</h3>
    <input type="number" name="student_id" placeholder="Student ID" required>
    <input type="file" name="profile_photo" accept="image/jpeg,image/png" required>
    <button type="submit">Upload Photo</button>
</form>
<?php

// =====================================================
// EXAMPLE 3: Upload Payment Receipt
// =====================================================

echo "\n=== Example 3: Upload Payment Receipt ===\n\n";

if (isset($_FILES['receipt'])) {
    $payment_id = $_POST['payment_id'] ?? null;
    
    if ($payment_id) {
        $result = $uploader->uploadPaymentReceipt(
            $_SESSION['academy_reference'],
            $payment_id,
            $_FILES['receipt']
        );
        
        if ($result['success']) {
            echo "✅ Receipt uploaded!\n";
            echo "File ID: {$result['file_id']}\n";
            echo "URL: {$result['url']}\n";
        } else {
            echo "❌ Upload failed: {$result['error']}\n";
        }
    }
}

// HTML Form for receipt upload
?>
<form method="post" enctype="multipart/form-data">
    <h3>Upload Payment Receipt</h3>
    <input type="number" name="payment_id" placeholder="Payment ID" required>
    <input type="file" name="receipt" accept=".pdf,image/jpeg,image/png" required>
    <button type="submit">Upload Receipt</button>
</form>
<?php

// =====================================================
// EXAMPLE 4: Upload Report
// =====================================================

echo "\n=== Example 4: Upload Report ===\n\n";

if (isset($_FILES['report'])) {
    $report_type = $_POST['report_type'] ?? 'general';
    
    $result = $uploader->uploadReport(
        $_SESSION['academy_reference'],
        $report_type,
        $_FILES['report'],
        [
            'description' => 'Monthly attendance report',
            'tags' => 'report,attendance,monthly'
        ]
    );
    
    if ($result['success']) {
        echo "✅ Report uploaded!\n";
        echo "File ID: {$result['file_id']}\n";
        echo "URL: {$result['url']}\n";
    } else {
        echo "❌ Upload failed: {$result['error']}\n";
    }
}

// HTML Form for report upload
?>
<form method="post" enctype="multipart/form-data">
    <h3>Upload Report</h3>
    <select name="report_type" required>
        <option value="attendance">Attendance Report</option>
        <option value="grades">Grades Report</option>
        <option value="financial">Financial Report</option>
        <option value="enrollment">Enrollment Report</option>
    </select>
    <input type="file" name="report" accept=".pdf,.xlsx" required>
    <button type="submit">Upload Report</button>
</form>
<?php

// =====================================================
// EXAMPLE 5: Validate File Before Upload
// =====================================================

echo "\n=== Example 5: Validate File ===\n\n";

if (isset($_FILES['file_to_validate'])) {
    $validation = $uploader->validateFile($_FILES['file_to_validate'], 'document');
    
    if ($validation['valid']) {
        echo "✅ File is valid!\n";
        echo "MIME type: {$validation['mime_type']}\n";
        echo "Extension: {$validation['extension']}\n";
    } else {
        echo "❌ File validation failed: {$validation['error']}\n";
    }
}

// =====================================================
// EXAMPLE 6: Sanitize Filename
// =====================================================

echo "\n=== Example 6: Sanitize Filename ===\n\n";

$unsafe_filenames = [
    'my document (copy) [final].pdf',
    'student photo @#$%.jpg',
    '../../../etc/passwd',
    'file with    spaces.docx'
];

foreach ($unsafe_filenames as $filename) {
    $sanitized = $uploader->sanitizeFilename($filename);
    echo "Original: {$filename}\n";
    echo "Sanitized: {$sanitized}\n\n";
}

// =====================================================
// EXAMPLE 7: Generate Unique Filename
// =====================================================

echo "\n=== Example 7: Generate Unique Filename ===\n\n";

$original = 'student_report.pdf';
for ($i = 0; $i < 3; $i++) {
    $unique = $uploader->generateUniqueFilename($original);
    echo "Unique filename {$i}: {$unique}\n";
}

// =====================================================
// EXAMPLE 8: Delete File
// =====================================================

echo "\n=== Example 8: Delete File ===\n\n";

if (isset($_POST['delete_file_id'])) {
    $file_path = $_POST['file_path'];
    
    $result = $uploader->deleteFile(
        $_SESSION['academy_reference'],
        $file_path
    );
    
    if ($result['success']) {
        echo "✅ File deleted successfully\n";
    } else {
        echo "❌ Delete failed: {$result['error']}\n";
    }
}

// =====================================================
// EXAMPLE 9: Complete Upload Handler
// =====================================================

/**
 * Complete example: Handle document upload with full error handling
 */
function handleDocumentUpload($uploader) {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        return;
    }
    
    if (!isset($_FILES['document'])) {
        return ['success' => false, 'error' => 'No file uploaded'];
    }
    
    // Check user is logged in
    if (!isset($_SESSION['user_id']) || !isset($_SESSION['academy_reference'])) {
        return ['success' => false, 'error' => 'Not authenticated'];
    }
    
    // Get metadata from form
    $metadata = [
        'description' => $_POST['description'] ?? null,
        'tags' => $_POST['tags'] ?? null,
        'related_id' => $_POST['related_id'] ?? null,
        'related_type' => $_POST['related_type'] ?? null
    ];
    
    // Upload file
    $result = $uploader->uploadDocument(
        $_SESSION['academy_reference'],
        $_FILES['document'],
        $metadata
    );
    
    // Return JSON response
    header('Content-Type: application/json');
    echo json_encode($result);
    exit;
}

// Usage:
// $result = handleDocumentUpload($uploader);

// =====================================================
// EXAMPLE 10: AJAX Upload (JavaScript)
// =====================================================
?>

<script>
// AJAX file upload with progress
async function uploadDocument(formData) {
    try {
        const response = await fetch('upload_handler.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            console.log('Upload successful!', result);
            alert(`File uploaded: ${result.filename}`);
            // Display file URL or thumbnail
            document.getElementById('file-url').href = result.url;
        } else {
            console.error('Upload failed:', result.error);
            alert(`Upload failed: ${result.error}`);
        }
    } catch (error) {
        console.error('Error:', error);
        alert('Upload error: ' + error.message);
    }
}

// Example usage with form
document.getElementById('upload-form').addEventListener('submit', async (e) => {
    e.preventDefault();
    
    const formData = new FormData(e.target);
    await uploadDocument(formData);
});

// Example with drag & drop
const dropZone = document.getElementById('drop-zone');

dropZone.addEventListener('dragover', (e) => {
    e.preventDefault();
    dropZone.classList.add('drag-over');
});

dropZone.addEventListener('drop', async (e) => {
    e.preventDefault();
    dropZone.classList.remove('drag-over');
    
    const files = e.dataTransfer.files;
    if (files.length > 0) {
        const formData = new FormData();
        formData.append('document', files[0]);
        formData.append('description', 'Drag and drop upload');
        
        await uploadDocument(formData);
    }
});
</script>

<?php

// =====================================================
// EXAMPLE 11: Batch Upload
// =====================================================

echo "\n=== Example 11: Batch Upload ===\n\n";

if (isset($_FILES['documents']) && is_array($_FILES['documents']['tmp_name'])) {
    $results = [];
    
    foreach ($_FILES['documents']['tmp_name'] as $index => $tmpName) {
        $file = [
            'name' => $_FILES['documents']['name'][$index],
            'type' => $_FILES['documents']['type'][$index],
            'tmp_name' => $tmpName,
            'error' => $_FILES['documents']['error'][$index],
            'size' => $_FILES['documents']['size'][$index]
        ];
        
        $result = $uploader->uploadDocument(
            $_SESSION['academy_reference'],
            $file,
            ['description' => 'Batch upload file ' . ($index + 1)]
        );
        
        $results[] = [
            'filename' => $file['name'],
            'success' => $result['success'],
            'error' => $result['error'] ?? null
        ];
    }
    
    echo "Uploaded " . count($results) . " files:\n";
    foreach ($results as $r) {
        $status = $r['success'] ? '✅' : '❌';
        echo "{$status} {$r['filename']}\n";
    }
}

// HTML Form for batch upload
?>
<form method="post" enctype="multipart/form-data">
    <h3>Batch Upload Documents</h3>
    <input type="file" name="documents[]" multiple accept=".pdf,.doc,.docx">
    <button type="submit">Upload All</button>
</form>

<?php

echo "\n=== Examples Complete ===\n";
echo "Check the code above for practical implementations.\n";

