# 📁 Tenant Directory Manager - Complete Guide

## 🎯 **Overview**

The `TenantDirectoryManager` class handles **initialization and management** of tenant directory structures in your multi-tenant education platform. It creates isolated, secure storage for each school/tenant with proper permissions and security files.

---

## ✨ **Features**

- ✅ **Complete Directory Initialization** - Creates full folder structure
- ✅ **Path Management** - Consistent path retrieval methods
- ✅ **Security Files** - Auto-creates .htaccess and index.php
- ✅ **Configuration Management** - JSON-based tenant settings
- ✅ **Directory Verification** - Health checks for all directories
- ✅ **Storage Tracking** - Calculate usage by category
- ✅ **Quota Management** - Per-tenant storage limits
- ✅ **Logging** - File and database logging

---

## 📂 **Directory Structure Created**

```
tenants/school_{tenant_id}/
├── uploads/
│   ├── documents/          # PDFs, Word, Excel files
│   ├── profile_photos/     # Student/staff photos
│   ├── payment_receipts/   # Payment confirmations
│   └── reports/            # Generated reports
├── backups/
│   ├── database/           # Database backups
│   └── files/              # File backups
├── logs/
│   ├── access/             # Access logs
│   ├── errors/             # Error logs
│   └── payments/           # Payment logs
├── temp/                   # Temporary files (auto-cleanup)
├── config.json             # Tenant configuration
├── .htaccess               # Security rules
└── index.php               # Prevent directory listing
```

---

## 🚀 **Quick Start**

### **Initialize Manager**

```php
<?php
require_once 'config.php';
require_once 'includes/tenant_directory_manager.php';

// With database connection (for logging)
$conn = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME, DB_USER, DB_PASS);
$manager = new TenantDirectoryManager($conn);

// Without database (file logging only)
$manager = new TenantDirectoryManager();
?>
```

### **Initialize New Tenant**

```php
$result = $manager->initializeTenant('soshigh_demo', 'SOS SOSHIGH School');

if ($result['success']) {
    echo "✅ Tenant initialized!";
    echo "Path: " . $result['path'];
} else {
    echo "❌ Error: " . $result['error'];
}
```

---

## 📋 **Core Methods**

### **1. initializeTenant()**

Creates complete directory structure for a new tenant.

```php
$result = $manager->initializeTenant(
    $tenant_id,       // Required: Unique identifier
    $tenant_name,     // Optional: Display name
    $custom_settings  // Optional: Custom configuration
);

// Returns:
[
    'success' => true,
    'message' => 'Tenant initialized successfully',
    'tenant_id' => 'soshigh_demo',
    'tenant_name' => 'SOS SOSHIGH School',
    'path' => '/path/to/tenants/school_soshigh_demo',
    'directories_created' => 10,
    'verification' => [...] // Verification results
]
```

**With Custom Settings:**

```php
$custom_settings = [
    'storage_quota' => 5368709120,  // 5GB
    'file_retention_days' => 730,   // 2 years
    'enable_auto_backup' => true,
    'backup_schedule' => 'daily'
];

$result = $manager->initializeTenant('premium_school', 'Premium Academy', $custom_settings);
```

---

### **2. getTenantRootPath()**

Get the root directory path for a tenant.

```php
$path = $manager->getTenantRootPath('soshigh_demo');
// Returns: /path/to/tenants/school_soshigh_demo
```

---

### **3. getTenantUploadPath()**

Get upload directory path for specific category.

```php
// Get documents path
$path = $manager->getTenantUploadPath('soshigh_demo', 'documents');
// Returns: /path/to/tenants/school_soshigh_demo/uploads/documents

// Valid categories:
// - documents
// - profile_photos
// - payment_receipts
// - reports
```

---

### **4. getTenantBackupPath()**

Get backup directory path.

```php
// Get base backup path
$path = $manager->getTenantBackupPath('soshigh_demo');
// Returns: /path/to/tenants/school_soshigh_demo/backups

// Get specific backup type
$path = $manager->getTenantBackupPath('soshigh_demo', 'database');
// Returns: /path/to/tenants/school_soshigh_demo/backups/database

// Valid types: database, files
```

---

### **5. getTenantLogPath()**

Get log directory path.

```php
// Get base log path
$path = $manager->getTenantLogPath('soshigh_demo');
// Returns: /path/to/tenants/school_soshigh_demo/logs

// Get specific log type
$path = $manager->getTenantLogPath('soshigh_demo', 'errors');
// Returns: /path/to/tenants/school_soshigh_demo/logs/errors

// Valid types: access, errors, payments
```

---

### **6. createTenantConfig()**

Create or update tenant configuration.

```php
$result = $manager->createTenantConfig('soshigh_demo', [
    'storage_quota' => 2147483648,  // 2GB
    'enable_notifications' => true,
    'notification_email' => 'admin@school.edu',
    'auto_backup_enabled' => true
]);

// Returns:
[
    'success' => true,
    'message' => 'Configuration created successfully',
    'path' => '/path/to/config.json',
    'config' => [...]  // Complete configuration
]
```

---

### **7. verifyTenantDirectories()**

Verify all directories exist and are writable.

```php
$result = $manager->verifyTenantDirectories('soshigh_demo');

// Returns:
[
    'success' => true,
    'tenant_id' => 'soshigh_demo',
    'root_path' => '/path/to/tenant',
    'directories_ok' => 11,
    'directories_missing' => 0,
    'directories_writable' => 11,
    'directories_not_writable' => 0,
    'details' => [
        'exists' => ['/', 'uploads/documents', ...],
        'missing' => [],
        'writable' => ['/', 'uploads/documents', ...],
        'not_writable' => []
    ],
    'message' => 'All directories verified'
]
```

---

### **8. calculateTenantStorageUsage()**

Calculate total storage usage for a tenant.

```php
$result = $manager->calculateTenantStorageUsage('soshigh_demo');

// Returns:
[
    'success' => true,
    'tenant_id' => 'soshigh_demo',
    'usage' => [
        'total' => 52428800,
        'uploads' => 41943040,
        'backups' => 8388608,
        'logs' => 2097152,
        'temp' => 0,
        'other' => 0
    ],
    'total_bytes' => 52428800,
    'total_formatted' => '50 MB',
    'quota_bytes' => 1073741824,
    'quota_formatted' => '1 GB',
    'percentage' => 4.88,
    'available_bytes' => 1021313024,
    'available_formatted' => '974 MB'
]
```

---

### **9. getTenantConfig()**

Get tenant configuration.

```php
$config = $manager->getTenantConfig('soshigh_demo');

// Returns array:
[
    'tenant_id' => 'soshigh_demo',
    'tenant_name' => 'SOS SOSHIGH School',
    'created_at' => '2025-01-30 10:00:00',
    'storage_quota' => 1073741824,
    'storage_used' => 0,
    'file_retention_days' => 365,
    ...
]
```

---

### **10. updateTenantQuota()**

Update tenant storage quota.

```php
$result = $manager->updateTenantQuota('soshigh_demo', 2147483648); // 2GB

if ($result['success']) {
    echo "✅ Quota updated to 2GB";
}
```

---

## 🔒 **Security Features**

### **Automatic Security Files**

The manager automatically creates:

#### **.htaccess Files**

**Root Directory:**
```apache
# Deny direct access to all files
Order Deny,Allow
Deny from all

# Allow images/PDFs through application
<Files ~ "\.(jpg|jpeg|png|gif|webp|pdf)$">
    Allow from all
</Files>
```

**Upload Directories:**
```apache
Options -Indexes
DirectoryIndex disabled

# Prevent PHP execution
php_flag engine off

# Allow safe file types
<FilesMatch "\.(jpg|jpeg|png|gif|webp|pdf|doc|docx|xls|xlsx)$">
    Order Allow,Deny
    Allow from all
</FilesMatch>
```

#### **index.php Files**

Placed in all directories to prevent listing:
```php
<?php
http_response_code(403);
die('Access denied');
```

---

## 🔧 **Practical Examples**

### **Example 1: School Creation Integration**

```php
<?php
function createNewSchool($school_data, $manager, $conn) {
    // 1. Insert into database
    $stmt = $conn->prepare("
        INSERT INTO tenants (academy_reference, name, email)
        VALUES (?, ?, ?)
    ");
    $stmt->execute([
        $school_data['academy_reference'],
        $school_data['name'],
        $school_data['email']
    ]);
    
    // 2. Initialize directories
    $result = $manager->initializeTenant(
        $school_data['academy_reference'],
        $school_data['name'],
        [
            'contact_email' => $school_data['email'],
            'storage_quota' => $school_data['storage_quota'] ?? 1073741824
        ]
    );
    
    if (!$result['success']) {
        throw new Exception("Failed to initialize: " . $result['error']);
    }
    
    // 3. Verify
    $verification = $manager->verifyTenantDirectories($school_data['academy_reference']);
    
    return [
        'success' => true,
        'tenant_id' => $school_data['academy_reference'],
        'directories' => $result['directories_created'],
        'verification' => $verification
    ];
}
?>
```

---

### **Example 2: Storage Monitoring**

```php
<?php
// Check storage for all tenants
$stmt = $conn->query("SELECT academy_reference FROM tenants");
$tenants = $stmt->fetchAll(PDO::FETCH_COLUMN);

foreach ($tenants as $tenant_id) {
    $usage = $manager->calculateTenantStorageUsage($tenant_id);
    
    if ($usage['percentage'] > 80) {
        // Send warning email
        echo "⚠️ {$tenant_id} is at {$usage['percentage']}% capacity\n";
        
        // Send notification
        sendStorageWarningEmail($tenant_id, $usage);
    }
}
?>
```

---

### **Example 3: Health Check Script**

```php
<?php
// Verify all tenant directories
$stmt = $conn->query("SELECT academy_reference, name FROM tenants");
$tenants = $stmt->fetchAll(PDO::FETCH_ASSOC);

$report = [];

foreach ($tenants as $tenant) {
    $verification = $manager->verifyTenantDirectories($tenant['academy_reference']);
    $storage = $manager->calculateTenantStorageUsage($tenant['academy_reference']);
    
    $report[] = [
        'name' => $tenant['name'],
        'status' => $verification['success'] ? 'OK' : 'ISSUES',
        'directories_ok' => $verification['directories_ok'],
        'directories_missing' => $verification['directories_missing'],
        'storage' => $storage['total_formatted'],
        'storage_percentage' => $storage['percentage']
    ];
}

// Display or email report
foreach ($report as $item) {
    $status_icon = $item['status'] === 'OK' ? '✅' : '❌';
    echo "{$status_icon} {$item['name']}\n";
    echo "  Directories: {$item['directories_ok']} OK";
    
    if ($item['directories_missing'] > 0) {
        echo ", {$item['directories_missing']} missing";
    }
    
    echo "\n  Storage: {$item['storage']} ({$item['storage_percentage']}%)\n\n";
}
?>
```

---

### **Example 4: Bulk Initialization**

```php
<?php
$schools = [
    ['id' => 'school_001', 'name' => 'School One', 'quota' => 1073741824],
    ['id' => 'school_002', 'name' => 'School Two', 'quota' => 2147483648],
    ['id' => 'school_003', 'name' => 'School Three', 'quota' => 5368709120]
];

foreach ($schools as $school) {
    $result = $manager->initializeTenant(
        $school['id'],
        $school['name'],
        ['storage_quota' => $school['quota']]
    );
    
    if ($result['success']) {
        echo "✅ {$school['name']} initialized\n";
    } else {
        echo "❌ {$school['name']} failed: {$result['error']}\n";
    }
}
?>
```

---

## ⚙️ **Configuration Options**

### **Default Configuration**

```json
{
    "tenant_id": "soshigh_demo",
    "tenant_name": "SOS SOSHIGH School",
    "created_at": "2025-01-30 10:00:00",
    "initialized_by": "admin",
    "storage_quota": 1073741824,
    "storage_used": 0,
    "file_retention_days": 365,
    "auto_cleanup_temp": true,
    "temp_cleanup_days": 7,
    "backup_retention_days": 30,
    "log_retention_days": 90
}
```

### **Custom Configuration**

```php
$custom_settings = [
    'storage_quota' => 5368709120,      // 5GB
    'file_retention_days' => 730,       // 2 years
    'temp_cleanup_days' => 3,
    'backup_retention_days' => 90,
    'log_retention_days' => 180,
    'enable_auto_backup' => true,
    'backup_schedule' => 'daily',
    'enable_notifications' => true,
    'notification_email' => 'admin@school.edu',
    'max_file_size' => 52428800,        // 50MB
    'allowed_file_types' => ['pdf', 'docx', 'jpg', 'png']
];
```

---

## 🐛 **Troubleshooting**

### **Issue: Permission Denied**

```
Error: Failed to create directory
```

**Solution:**
```bash
# Set proper permissions on tenants directory
chmod 755 tenants/
chown www-data:www-data tenants/

# Or set more permissive (if above doesn't work)
chmod 777 tenants/
```

---

### **Issue: Directory Already Exists**

```
Message: Tenant already initialized
```

**This is not an error!** The system detects existing installations and skips re-initialization.

---

### **Issue: Invalid Tenant ID**

```
Error: Invalid tenant ID format
```

**Valid formats:**
- ✅ `soshigh_demo`
- ✅ `school-001`
- ✅ `waterford2025`
- ❌ `school@123` (no special chars except - and _)
- ❌ `ab` (minimum 3 characters)

---

## 📊 **Permissions Reference**

| Item | Permission | Numeric |
|------|-----------|---------|
| Directories (standard) | `rwxr-xr-x` | `0755` |
| Upload directories | `rwxrwxrwx` | `0777` |
| Configuration files | `rw-r--r--` | `0644` |
| Security files | `rw-r--r--` | `0644` |

---

## 🎯 **Best Practices**

1. ✅ **Initialize on School Creation**
   - Always initialize directories when creating a new school
   - Verify initialization before proceeding

2. ✅ **Regular Health Checks**
   - Run verification monthly
   - Check storage usage weekly
   - Monitor quota usage

3. ✅ **Storage Monitoring**
   - Send warnings at 80% capacity
   - Block uploads at 95% capacity
   - Auto-cleanup temp files

4. ✅ **Backup Strategy**
   - Use dedicated backup directory
   - Separate database and file backups
   - Implement retention policies

5. ✅ **Logging**
   - Use separate logs per type (access, errors, payments)
   - Rotate logs regularly
   - Clean old logs per retention policy

---

## ✅ **Summary**

The `TenantDirectoryManager` provides:

- ✅ **Complete directory initialization**
- ✅ **Consistent path management**
- ✅ **Automatic security files**
- ✅ **Configuration management**
- ✅ **Storage tracking**
- ✅ **Health verification**
- ✅ **Production-ready code**

**Perfect for multi-tenant education platforms!** 🚀

---

*File: TENANT_DIRECTORY_MANAGER_GUIDE.md*  
*Last Updated: 2025*  
*Version: 1.0*

